!! Copyright (C) 2009 X. Andrade
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module hamiltonian_elec_base_oct_m
  use accel_oct_m
  use batch_oct_m
  use batch_ops_oct_m
  use blas_oct_m
  use comm_oct_m
  use debug_oct_m
  use derivatives_oct_m
  use electron_space_oct_m
  use epot_oct_m
  use global_oct_m
  use hardware_oct_m
  use math_oct_m
  use mesh_oct_m
  use messages_oct_m
  use mpi_oct_m
  use nl_operator_oct_m
  use profiling_oct_m
  use ps_oct_m
  use space_oct_m
  use states_elec_oct_m
  use states_elec_dim_oct_m
  use types_oct_m
  use wfs_elec_oct_m

  implicit none

  private

  public ::                                         &
    hamiltonian_elec_base_t,                        &
    dhamiltonian_elec_base_local_sub,               &
    zhamiltonian_elec_base_local_sub

  !> @brief The basic Hamiltonian for electronic system
  !!
  !! This object stores and applies an electromagnetic potential that
  !! can be represented by different types of potentials.
  !!
  !! Different components of scalar or vector potentials are added together
  !! as for the application of the Hamiltonian onlt the sum matters.
  !!
  type hamiltonian_elec_base_t
    private
    integer                                       :: nspin !< number of spin channels
    real(real64)                                  :: mass  !< Needed to compute the magnetic terms, if the mass is not one.
    real(real64)                                  :: rashba_coupling !< Rashba coupling strength
    type(nl_operator_t),      pointer,     public :: kinetic !< kinetic energy operator
    real(real64),             allocatable, public :: potential(:, :)           !< real scalar potential
    real(real64),             allocatable, public :: Impotential(:, :)         !< imaginary scalar potential
    real(real64),             allocatable, public :: uniform_magnetic_field(:) !< uniform magnetic field
    !!                                                                            (assumed to be in Gaussian units)
    real(real64),             allocatable, public :: magnetic_field(:, :)      !< non-uniform magnetic field
    !!                                                                            (assumed to be in Gaussian units)
    real(real64),             allocatable, public :: zeeman_pot(:, :)
    real(real64),             allocatable, public :: uniform_vector_potential(:) !< @brief a uniform vector potential
    !!
    !!                                                                              in some cases, absorbed in the vector_potential
    !!                                                                              i.e. the non-uniform vector potential
    real(real64),             allocatable, public :: vector_potential(:, :)      !< general (non-uniform) vector potential
    type(accel_mem_t)                             :: potential_accel
    type(accel_mem_t)                             :: impotential_accel
  contains

    procedure :: init => hamiltonian_elec_base_init
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_init
    procedure :: end => hamiltonian_elec_base_end
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_end
    procedure :: clear => hamiltonian_elec_base_clear
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_clear
    procedure :: update_magnetic_terms => hamiltonian_elec_base_update_magnetic_terms
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_update
    procedure :: allocate_field => hamiltonian_elec_base_allocate
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_allocate
    procedure :: accel_copy_pot => hamiltonian_elec_base_accel_copy_pot
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_accel_copy_pot
    procedure :: has_magnetic => hamiltonian_elec_base_has_magnetic
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_has_magnetic
    procedure :: has_zeeman => hamiltonian_elec_base_has_zeeman
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_has_zeeman
    procedure :: has_vector_potential => hamiltonian_elec_base_has_vector_potential
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_has_vector_potential
    procedure :: calc_rashba => hamiltonian_elec_base_rashba
    !< @copydoc hamiltonian_elec_base_oct_m::hamiltonian_elec_base_rashba
    procedure :: dcalc_magnetic => dhamiltonian_elec_base_magnetic
    !< @copydoc hamiltonian_elec_base_oct_m::dhamiltonian_elec_base_magnetic
    procedure :: zcalc_magnetic => zhamiltonian_elec_base_magnetic
    !< @copydoc hamiltonian_elec_base_oct_m::zhamiltonian_elec_base_magnetic
    procedure :: dcalc_local => dhamiltonian_elec_base_local
    !< @copydoc hamiltonian_elec_base_oct_m::dhamiltonian_elec_base_local
    procedure :: zcalc_local => zhamiltonian_elec_base_local
    !< @copydoc hamiltonian_elec_base_oct_m::zhamiltonian_elec_base_local
  end type hamiltonian_elec_base_t

  integer, parameter, public ::          &
    TERM_ALL                 = HUGE(1),  &
    TERM_KINETIC             =   1,      &
    TERM_LOCAL_POTENTIAL     =   2,      &
    TERM_NON_LOCAL_POTENTIAL =   4,      &
    TERM_OTHERS              =   8,      &
    TERM_LOCAL_EXTERNAL      =  16,      &
    TERM_MGGA                =  32,      &
    TERM_DFT_U               =  64,      &
    TERM_RDMFT_OCC           = 128

  integer, parameter, public ::            &
    FIELD_POTENTIAL                = 1,    &
    FIELD_VECTOR_POTENTIAL         = 2,    &
    FIELD_UNIFORM_VECTOR_POTENTIAL = 4,    &
    FIELD_UNIFORM_MAGNETIC_FIELD   = 8,    &
    FIELD_MAGNETIC_FIELD           = 16

contains

  ! ---------------------------------------------------------
  !> initialize the hamiltonian_elec_base_t object
  !!
  subroutine hamiltonian_elec_base_init(this, nspin, mass, rashba_coupling)
    class(hamiltonian_elec_base_t), intent(inout) :: this
    integer,                  intent(in)    :: nspin
    real(real64),             intent(in)    :: mass
    real(real64),             intent(in)    :: rashba_coupling

    PUSH_SUB(hamiltonian_elec_base_init)

    this%nspin = nspin
    this%mass  = mass
    this%rashba_coupling = rashba_coupling

    POP_SUB(hamiltonian_elec_base_init)
  end subroutine hamiltonian_elec_base_init

  ! ---------------------------------------------------------
  !> Finalizer for hamiltonian_elec_base_t
  !!
  subroutine hamiltonian_elec_base_end(this)
    class(hamiltonian_elec_base_t), intent(inout) :: this

    PUSH_SUB(hamiltonian_elec_base_end)

    if (allocated(this%potential) .and. accel_is_enabled()) then
      call accel_release_buffer(this%potential_accel)
      if (allocated(this%Impotential)) then
        call accel_release_buffer(this%impotential_accel)
      end if
    end if

    SAFE_DEALLOCATE_A(this%potential)
    SAFE_DEALLOCATE_A(this%Impotential)
    SAFE_DEALLOCATE_A(this%vector_potential)
    SAFE_DEALLOCATE_A(this%uniform_vector_potential)
    SAFE_DEALLOCATE_A(this%uniform_magnetic_field)
    SAFE_DEALLOCATE_A(this%magnetic_field)
    SAFE_DEALLOCATE_A(this%zeeman_pot)

    POP_SUB(hamiltonian_elec_base_end)
  end subroutine hamiltonian_elec_base_end

  ! ----------------------------------------------------------
  !
  !> This functions sets to zero all fields that are currently
  !! allocated.
  !
  subroutine hamiltonian_elec_base_clear(this, np)
    class(hamiltonian_elec_base_t), intent(inout) :: this
    integer,                       intent(in)    :: np

    integer :: ip, ispin

    PUSH_SUB(hamiltonian_elec_clear)

    if (allocated(this%potential)) then
      do ispin = 1, this%nspin
        !$omp parallel do schedule(static)
        do ip = 1, np
          this%potential(ip, ispin) = M_ZERO
        end do
      end do
    end if

    if (allocated(this%Impotential)) then
      do ispin = 1, this%nspin
        !$omp parallel do schedule(static)
        do ip = 1, np
          this%Impotential(ip, ispin) = M_ZERO
        end do
      end do
    end if

    if (allocated(this%uniform_vector_potential)) then
      this%uniform_vector_potential = M_ZERO
    end if

    if (allocated(this%vector_potential)) then
      this%vector_potential = M_ZERO
    end if

    if (allocated(this%uniform_magnetic_field)) then
      this%uniform_magnetic_field = M_ZERO
    end if

    if (allocated(this%magnetic_field)) then
      this%magnetic_field = M_ZERO
    end if

    if (allocated(this%zeeman_pot)) then
      this%zeeman_pot = M_ZERO
    end if

    POP_SUB(hamiltonian_elec_clear)
  end subroutine hamiltonian_elec_base_clear


  ! ---------------------------------------------------------------
  !> This function ensures that the corresponding field is allocated.
  !
  subroutine hamiltonian_elec_base_allocate(this, mesh, field, complex_potential)
    class(hamiltonian_elec_base_t), intent(inout) :: this
    class(mesh_t),            intent(in)    :: mesh
    integer,                  intent(in)    :: field
    logical,                  intent(in)    :: complex_potential

    integer :: ip, ispin

    PUSH_SUB(hamiltonian_elec_base_allocate)

    if (bitand(FIELD_POTENTIAL, field) /= 0) then
      if (.not. allocated(this%potential)) then
        SAFE_ALLOCATE(this%potential(1:mesh%np, 1:this%nspin))
        do ispin = 1, this%nspin
          !$omp parallel do schedule(static)
          do ip = 1, mesh%np
            this%potential(ip, ispin) = M_ZERO
          end do
        end do
        if (complex_potential) then
          SAFE_ALLOCATE(this%Impotential(1:mesh%np, 1:this%nspin))
          do ispin = 1, this%nspin
            !$omp parallel do schedule(static)
            do ip = 1, mesh%np
              this%Impotential(ip, ispin) = M_ZERO
            end do
          end do
        end if
        if (accel_is_enabled()) then
          call accel_create_buffer(this%potential_accel, ACCEL_MEM_READ_ONLY, TYPE_FLOAT, accel_padded_size(mesh%np)*this%nspin)
          if (complex_potential) then
            call accel_create_buffer(this%impotential_accel, ACCEL_MEM_READ_ONLY, TYPE_FLOAT, &
              accel_padded_size(mesh%np)*this%nspin)
          end if
        end if
      end if
    end if

    if (bitand(FIELD_UNIFORM_VECTOR_POTENTIAL, field) /= 0) then
      if (.not. allocated(this%uniform_vector_potential)) then
        SAFE_ALLOCATE(this%uniform_vector_potential(1:mesh%box%dim))
        this%uniform_vector_potential = M_ZERO
      end if
    end if

    if (bitand(FIELD_VECTOR_POTENTIAL, field) /= 0) then
      if (.not. allocated(this%vector_potential)) then
        SAFE_ALLOCATE(this%vector_potential(1:mesh%box%dim, 1:mesh%np))
        this%vector_potential = M_ZERO
      end if
    end if

    if (bitand(FIELD_UNIFORM_MAGNETIC_FIELD, field) /= 0) then
      if (.not. allocated(this%uniform_magnetic_field)) then
        SAFE_ALLOCATE(this%uniform_magnetic_field(1:max(mesh%box%dim, 3)))
        this%uniform_magnetic_field = M_ZERO
      end if
    end if

    if ((bitand(FIELD_MAGNETIC_FIELD, field) /= 0) .or. &
      bitand(FIELD_UNIFORM_MAGNETIC_FIELD, field) /= 0) then
      if (.not. allocated(this%magnetic_field)) then
        SAFE_ALLOCATE(this%magnetic_field(1:mesh%np, 1:max(mesh%box%dim, 3)))
        this%magnetic_field = M_ZERO
        SAFE_ALLOCATE(this%zeeman_pot(1:mesh%np, 1:this%nspin))
        this%zeeman_pot = M_ZERO
      end if
    end if

    POP_SUB(hamiltonian_elec_base_allocate)
  end subroutine hamiltonian_elec_base_allocate

  ! ----------------------------------------------------------
  !
  !> @brief update the magnetic terms of the hamiltonian_elec_base_t.
  !!
  !! If both a uniform and non-uniform vector potentials are allocated,
  !! this function copies the uniform in the non-uniform one. In the
  !! future it may perform other internal consistency operations.
  !
  subroutine hamiltonian_elec_base_update_magnetic_terms(this, mesh, gyromagnetic_ratio, ispin)
    class(hamiltonian_elec_base_t), intent(inout) :: this
    class(mesh_t),                  intent(in)    :: mesh
    real(real64),                   intent(in)    :: gyromagnetic_ratio
    integer,                        intent(in)    :: ispin

    real(real64) :: b_norm2, cc
    integer :: idir, ip

    PUSH_SUB(hamiltonian_elec_base_update_magnetic_terms)

    if (allocated(this%uniform_vector_potential) .and. allocated(this%vector_potential)) then
      ! copy the uniform vector potential onto the non-uniform one
      do idir = 1, mesh%box%dim
        !$omp parallel do schedule(static)
        do ip = 1, mesh%np
          this%vector_potential(idir, ip) = &
            this%vector_potential(idir, ip) + this%uniform_vector_potential(idir)
        end do
      end do
      SAFE_DEALLOCATE_A(this%uniform_vector_potential)
    end if


    if (.not. allocated(this%magnetic_field)) then
      POP_SUB(hamiltonian_elec_base_update_magnetic_terms)
      return
    end if

    ! add the Zeeman term
    ! It reads in the Pauli equation as  $\frac{|e|\hbar}{2m} \sigma\cdot\vec{B}$
    ! The gyromagnetic ratio is twice the magnetic moment of the electron in units of Bohr magneton
    ! This is why we multiply by $\frac{|g_e|}{2}$.
    ! The factor 1/c comes from the factor 1/c in front of the vector potential, which leads to
    ! the Zeeman term starting from the Dirac Hamiltonian.
    cc = M_HALF/P_C*gyromagnetic_ratio*M_HALF

    if (allocated(this%uniform_magnetic_field) ) then
      !$omp parallel private(ip)
      do idir = 1, 3
        !$omp do
        do ip = 1, mesh%np
          this%magnetic_field(ip, idir) = this%magnetic_field(ip, idir) + this%uniform_magnetic_field(idir)
        end do
        !$omp end do nowait
      end do
      !$omp end parallel
    end if

    select case (ispin)
    case (SPIN_POLARIZED)
      !$omp parallel do private(b_norm2)
      do ip = 1, mesh%np
        b_norm2 = norm2(this%magnetic_field(ip, 1:max(mesh%box%dim, 3)))
        this%zeeman_pot(ip, 1) =   cc*b_norm2
        this%zeeman_pot(ip, 2) = - cc*b_norm2
      end do

    case (SPINORS)
      !$omp parallel do
      do ip = 1, mesh%np
        this%zeeman_pot(ip, 1) =  cc*this%magnetic_field(ip, 3)
        this%zeeman_pot(ip, 2) = - cc*this%magnetic_field(ip, 3)
        this%zeeman_pot(ip, 3) =   cc*this%magnetic_field(ip, 1)
        this%zeeman_pot(ip, 4) = - cc*this%magnetic_field(ip, 2)
      end do
    end select
    SAFE_DEALLOCATE_A(this%uniform_magnetic_field)

    POP_SUB(hamiltonian_elec_base_update_magnetic_terms)
  end subroutine hamiltonian_elec_base_update_magnetic_terms


  !--------------------------------------------------------
  !> copy the potential to the acceleration device buffer
  !
  subroutine hamiltonian_elec_base_accel_copy_pot(this, mesh)
    class(hamiltonian_elec_base_t), intent(inout) :: this
    class(mesh_t),            intent(in)    :: mesh

    integer :: offset, ispin

    PUSH_SUB(hamiltonian_elec_base_accel_copy_pot)

    if (allocated(this%potential) .and. accel_is_enabled()) then
      offset = 0
      do ispin = 1, this%nspin
        call accel_write_buffer(this%potential_accel, mesh%np, this%potential(:, ispin), offset = offset)
        if(allocated(this%Impotential)) then
          call accel_write_buffer(this%impotential_accel, mesh%np, this%Impotential(:, ispin), offset = offset)
        end if
        offset = offset + accel_padded_size(mesh%np)
      end do
    end if

    POP_SUB(hamiltonian_elec_base_accel_copy_pot)
  end subroutine hamiltonian_elec_base_accel_copy_pot

  ! ----------------------------------------------------------------------------------
  !> return .true. if the Hamiltonian contains any magnetic field
  !
  logical pure function hamiltonian_elec_base_has_magnetic(this) result(has_magnetic)
    class(hamiltonian_elec_base_t), intent(in) :: this

    has_magnetic = allocated(this%vector_potential) &
      .or. allocated(this%uniform_magnetic_field)

  end function hamiltonian_elec_base_has_magnetic

  ! ----------------------------------------------------------------------------------
  !> return .true. of the Hamiltonian contains a zeeman term
  !
  logical pure function hamiltonian_elec_base_has_zeeman(this) result(has_zeeman)
    class(hamiltonian_elec_base_t), intent(in) :: this

    has_zeeman = allocated(this%zeeman_pot)

  end function hamiltonian_elec_base_has_zeeman

  ! ----------------------------------------------------------------------------------
  !> return .true. of the Hamiltonian contains any vector potential
  !
  logical pure function hamiltonian_elec_base_has_vector_potential(this) result(has_vector_potential)
    class(hamiltonian_elec_base_t), intent(in) :: this

    has_vector_potential = allocated(this%vector_potential) &
      .or. allocated(this%uniform_vector_potential)

  end function hamiltonian_elec_base_has_vector_potential

  ! ---------------------------------------------------------------------------------------
  subroutine hamiltonian_elec_base_rashba(this, mesh, der, std, psib, vpsib)
    class(hamiltonian_elec_base_t),  intent(in)    :: this
    class(mesh_t),                  intent(in)    :: mesh
    type(derivatives_t),            intent(in)    :: der
    type(states_elec_dim_t),        intent(in)    :: std
    type(wfs_elec_t), target,       intent(in)    :: psib
    type(wfs_elec_t), target,       intent(inout) :: vpsib

    integer :: ist, idim, ip
    complex(real64), allocatable :: psi(:, :), vpsi(:, :), grad(:, :, :)

    PUSH_SUB(hamiltonian_elec_base_rashba)

    if (abs(this%rashba_coupling) < M_EPSILON) then
      POP_SUB(hamiltonian_elec_base_rashba)
      return
    end if
    ASSERT(std%ispin == SPINORS)
    ASSERT(mesh%box%dim == 2)
    ASSERT(psib%type() == TYPE_CMPLX)
    ASSERT(vpsib%type() == TYPE_CMPLX)

    SAFE_ALLOCATE(psi(1:mesh%np_part, 1:std%dim))
    SAFE_ALLOCATE(vpsi(1:mesh%np, 1:std%dim))
    SAFE_ALLOCATE(grad(1:mesh%np, 1:mesh%box%dim, 1:std%dim))

    do ist = 1, psib%nst
      call batch_get_state(psib, ist, mesh%np_part, psi)
      call batch_get_state(vpsib, ist, mesh%np, vpsi)

      do idim = 1, std%dim
        call zderivatives_grad(der, psi(:, idim), grad(:, :, idim), ghost_update = .false., set_bc = .false.)
      end do

      if (allocated(this%vector_potential)) then
        do ip = 1, mesh%np
          vpsi(ip, 1) = vpsi(ip, 1) - &
            (this%rashba_coupling) * cmplx(this%vector_potential(2, ip), this%vector_potential(1, ip), real64) * psi(ip, 2)
          vpsi(ip, 2) = vpsi(ip, 2) - &
            (this%rashba_coupling) * cmplx(this%vector_potential(2, ip), -this%vector_potential(1, ip), real64) * psi(ip, 1)
        end do
      end if

      do ip = 1, mesh%np
        vpsi(ip, 1) = vpsi(ip, 1) - &
          this%rashba_coupling*( grad(ip, 1, 2) - M_zI*grad(ip, 2, 2))
        vpsi(ip, 2) = vpsi(ip, 2) + &
          this%rashba_coupling*( grad(ip, 1, 1) + M_zI*grad(ip, 2, 1))
      end do

      call batch_set_state(vpsib, ist, mesh%np, vpsi)
    end do

    SAFE_DEALLOCATE_A(grad)
    SAFE_DEALLOCATE_A(vpsi)
    SAFE_DEALLOCATE_A(psi)

    POP_SUB(hamiltonian_elec_base_rashba)
  end subroutine hamiltonian_elec_base_rashba

#include "undef.F90"
#include "real.F90"
#include "hamiltonian_elec_base_inc.F90"

#include "undef.F90"
#include "complex.F90"
#include "hamiltonian_elec_base_inc.F90"

end module hamiltonian_elec_base_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
