!! Copyright (C) 2016-2019 N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

!>@brief Apply the +U nonlocal potential to psib and adds the result to hpsib
subroutine X(lda_u_apply)(this, d, mesh, psib, hpsib)
  type(lda_u_t),           intent(in)    :: this
  class(mesh_t),           intent(in)    :: mesh
  type(wfs_elec_t),        intent(in)    :: psib
  type(wfs_elec_t),        intent(inout) :: hpsib
  type(states_elec_dim_t), intent(in)    :: d

  integer :: ios
  R_TYPE, allocatable :: phi_dot_psi(:,:,:,:), weight(:,:,:), psi(:,:)
  type(orbitalset_t), pointer  :: os

  call profiling_in(TOSTRING(X(DFTU_APPLY)))

  PUSH_SUB(lda_u_apply)

  SAFE_ALLOCATE(weight(1:this%maxnorbs, 1:psib%nst_linear, 1:this%norbsets))
  SAFE_ALLOCATE(phi_dot_psi(1:d%dim, 1:this%maxnorbs, 1:psib%nst, 1:this%norbsets))
  SAFE_ALLOCATE(psi(1:mesh%np, 1:d%dim))

  ! We have to compute
  ! hpsi> += sum_m |phi m> sum_mp Vmmp <phi mp | psi >
  !
  ! We first compute <phi m | psi> for all orbitals of the atom
  !
  do ios = 1, this%norbsets
    os => this%orbsets(ios)
    call X(orbitalset_get_coeff_batch)(os, d%dim, psib, phi_dot_psi(:,:,:, ios), reduce=.false.)
  end do
  if (mesh%parallel_in_domains) then
    call profiling_in(TOSTRING(X(DFT_U_APPLY_REDUCE)))
    call mesh%allreduce(phi_dot_psi)
    call profiling_out(TOSTRING(X(DFT_U_APPLY_REDUCE)))
  end if

  weight(:,:,:) = R_TOTYPE(M_ZERO)
  !
  do ios = 1, this%norbsets
    ! We now compute weight = sum_mp Vmmp <phi mp | psi >
    os => this%orbsets(ios)
    !
    call X(dft_u_get_weight_projectors)(this, os, ios, d%ispin, psib, phi_dot_psi, weight(:, :, ios))
  end do !ios

  !We add the orbitals properly weighted to hpsi
  do ios = 1, this%norbsets
    os => this%orbsets(ios)
    call X(orbitalset_add_to_batch)(os, d%dim, hpsib, weight(:,:,ios))
  end do

  SAFE_DEALLOCATE_A(psi)
  SAFE_DEALLOCATE_A(phi_dot_psi)
  SAFE_DEALLOCATE_A(weight)

  POP_SUB(lda_u_apply)
  call profiling_out(TOSTRING(X(DFTU_APPLY)))

end subroutine X(lda_u_apply)

! ---------------------------------------------------------
!>@brief This routine computes the values of the occupation matrices
subroutine X(update_occ_matrices)(this, namespace, mesh, st, lda_u_energy, phase)
  type(lda_u_t),                        intent(inout) :: this
  type(namespace_t),                    intent(in)    :: namespace
  class(mesh_t),                        intent(in)    :: mesh
  type(states_elec_t),          target, intent(inout) :: st
  real(real64),                         intent(inout) :: lda_u_energy
  type(phase_t),                 optional, intent(in) :: phase

  integer :: ios, im, ik, ist, ispin, norbs, idim, inn, im2, ios2, ib, ibatch, imp
  R_TYPE, allocatable :: psi(:,:)
  R_TYPE, allocatable :: phi_dot_psi(:,:,:,:)
  real(real64)   :: weight, renorm_weight
  R_TYPE  :: weight_phase, renorm_weight_phase
  type(orbitalset_t), pointer :: os, os2
  real(real64), allocatable :: muliken_charge(:)
  type(wfs_elec_t), pointer :: epsib
  real(real64) :: num, denom, nsigma

  call profiling_in(TOSTRING(X(DFTU_OCC_MATRICES)))

  PUSH_SUB(update_occ_matrices)

  this%X(n)(1:this%maxnorbs, 1:this%maxnorbs, 1:st%d%nspin, 1:this%norbsets) = R_TOTYPE(M_ZERO)
  SAFE_ALLOCATE(psi(1:mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(phi_dot_psi(1:st%d%dim, 1:this%maxnorbs, 1:st%block_size, 1:this%norbsets))
  if (this%level == DFT_U_ACBN0) then
    this%X(n_alt)(1:this%maxnorbs, 1:this%maxnorbs, 1:st%d%nspin, 1:this%norbsets) = R_TOTYPE(M_ZERO)
    if (this%intersite) then
      this%X(n_ij)(1:this%maxnorbs, 1:this%maxnorbs, 1:st%d%nspin, &
        1:this%norbsets, 1:this%maxneighbors) = R_TOTYPE(M_ZERO)
      this%X(n_alt_ij)(1:this%maxnorbs, 1:this%maxnorbs, 1:st%d%nspin, &
        1:this%norbsets, 1:this%maxneighbors) = R_TOTYPE(M_ZERO)
      this%X(n_alt_ii)(1:2, 1:this%maxnorbs, 1:st%d%nspin, &
        1:this%norbsets, 1:this%maxneighbors) = R_TOTYPE(M_ZERO)
    end if
    SAFE_ALLOCATE(muliken_charge(this%norbsets))
  end if

  !TODO: use symmetries of the occupation matrices
  do ik = st%d%kpt%start, st%d%kpt%end
    ispin =  st%d%get_spin_index(ik)

    do ib = st%group%block_start, st%group%block_end

      call st%group%psib(ib, ik)%do_pack(copy = .true.)
      if (present(phase)) then
        SAFE_ALLOCATE(epsib)
        call st%group%psib(ib, ik)%copy_to(epsib)
        call phase%apply_to(mesh, mesh%np, .false., epsib, src = st%group%psib(ib, ik))
      else
        epsib => st%group%psib(ib, ik)
      end if

      do ios = 1, this%norbsets
        os => this%orbsets(ios)
        !We first compute the matrix elemets <orb_m |\psi>
        !taking into account phase correction if needed
        call X(orbitalset_get_coeff_batch)(os, st%d%dim, epsib, phi_dot_psi(:, :, :, ios))
      end do !ios

      do ibatch = 1, epsib%nst
        ist = epsib%ist(ibatch)

        weight = st%kweights(ik)*st%occ(ist, ik)
        if (weight < 1.0e-10_real64) cycle

        !We compute the on-site occupation of the site, if needed
        if (this%level == DFT_U_ACBN0) then
          this%renorm_occ(:,:,:,ist,ik) = M_ONE-this%acbn0_screening
          muliken_charge = M_ZERO
          do ios = 1, this%norbsets
            os => this%orbsets(ios)

            do im = 1, os%norbs
              do idim = 1, st%d%dim
                muliken_charge(ios) = muliken_charge(ios) + abs(phi_dot_psi(idim,im,ibatch,ios))**2  &
                  * this%acbn0_screening
              end do
            end do

            this%renorm_occ(os%spec_index,os%nn,os%ll,ist,ik) = &
              this%renorm_occ(os%spec_index,os%nn,os%ll,ist,ik) + muliken_charge(ios)
          end do

        end if


        !We can compute the (renormalized) occupation matrices
        do ios = 1, this%norbsets
          os => this%orbsets(ios)
          norbs = os%norbs

          do im = 1, norbs

            if (st%d%ispin /= SPINORS) then !Collinear case
              this%X(n)(1:norbs, im, ispin, ios) = this%X(n)(1:norbs, im, ispin, ios) &
                + weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))
              !We compute the renomalized occupation matrices
              if (this%level == DFT_U_ACBN0) then
                renorm_weight = this%renorm_occ(os%spec_index, os%nn, os%ll, ist, ik) * weight
                this%X(n_alt)(1:norbs, im, ispin, ios) = this%X(n_alt)(1:norbs, im, ispin, ios) &
                  + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))

                !Generalized occupation matrices
                if (this%intersite) then

                  do inn = 1, os%nneighbors
                    ios2 = os%map_os(inn)
                    os2 => this%orbsets(ios2)

                    renorm_weight = (this%renorm_occ(os%spec_index, os%nn, os%ll, ist, ik) + &
                      this%renorm_occ(os2%spec_index, os2%nn, os2%ll, ist, ik)) * weight
                    if (os%spec_index == os2%spec_index) then
                      renorm_weight = M_HALF* renorm_weight
                    end if

                    this%X(n_alt_ii)(1, im, ispin, ios, inn) = this%X(n_alt_ii)(1, im, ispin, ios, inn) &
                      + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))

                    weight_phase = weight
                    renorm_weight_phase = renorm_weight
                    if (present(phase)) then
#ifdef R_TCOMPLEX
                      weight_phase = weight_phase * os%phase_shift(inn, ik)
                      renorm_weight_phase = renorm_weight_phase * os%phase_shift(inn, ik)
#else
                      ! Phase can only by applied when having complex wavefunctions
                      ASSERT(.false.)
#endif
                    end if


                    do im2 = 1, os2%norbs

                      if (im == 1) then
                        this%X(n_alt_ii)(2, im2, ispin, ios, inn) = this%X(n_alt_ii)(2, im2, ispin, ios, inn) &
                          + renorm_weight * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im2, ibatch, ios2))
                      end if

                      this%X(n_ij)(im, im2, ispin, ios, inn) = this%X(n_ij)(im, im2, ispin, ios, inn) &
                        + weight_phase * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                      this%X(n_alt_ij)(im, im2, ispin, ios, inn) = this%X(n_alt_ij)(im, im2, ispin, ios, inn) &
                        + renorm_weight_phase * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                    end do !im2
                  end do !inn
                end if
              end if

            else !Noncollinear case

              this%X(n)(1:norbs, im, 1, ios) = this%X(n)(1:norbs, im, 1, ios) &
                + weight*phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))
              this%X(n)(1:norbs, im, 2, ios) = this%X(n)(1:norbs, im, 2, ios) &
                + weight*phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, 1:norbs, ibatch, ios))
              this%X(n)(1:norbs, im, 3, ios) = this%X(n)(1:norbs, im, 3, ios) &
                + weight*phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, 1:norbs, ibatch, ios))
              this%X(n)(1:norbs, im, 4, ios) = this%X(n)(1:norbs, im, 4, ios) &
                + weight*phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))
              !We compute the renomalized occupation matrices
              if (this%level == DFT_U_ACBN0) then
                renorm_weight = this%renorm_occ(os%spec_index,os%nn,os%ll,ist,ik)*weight
                this%X(n_alt)(1:norbs,im,1,ios) = this%X(n_alt)(1:norbs, im, 1, ios) &
                  + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))
                this%X(n_alt)(1:norbs,im,2,ios) = this%X(n_alt)(1:norbs, im, 2, ios) &
                  + renorm_weight * phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, 1:norbs, ibatch, ios))
                this%X(n_alt)(1:norbs,im,3,ios) = this%X(n_alt)(1:norbs, im, 3, ios) &
                  + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, 1:norbs, ibatch, ios))
                this%X(n_alt)(1:norbs,im,4,ios) = this%X(n_alt)(1:norbs, im, 4, ios) &
                  + renorm_weight * phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, 1:norbs, ibatch, ios))

                if (this%intersite) then
                  do inn = 1, os%nneighbors
                    ios2 = os%map_os(inn)
                    os2 => this%orbsets(ios2)

                    renorm_weight = (this%renorm_occ(os%spec_index, os%nn, os%ll, ist, ik) + &
                      this%renorm_occ(os2%spec_index, os2%nn, os2%ll, ist, ik) ) * weight
                    if (os%spec_index == os2%spec_index) then
                      renorm_weight = M_HALF * renorm_weight
                    end if

                    weight_phase = weight
                    renorm_weight_phase = renorm_weight
                    if (present(phase)) then
#ifdef R_TCOMPLEX
                      weight_phase = weight_phase * os%phase_shift(inn, ik)
                      renorm_weight_phase = renorm_weight_phase * os%phase_shift(inn, ik)
#else
                      ! Phase can only by applied when having complex wavefunctions
                      ASSERT(.false.)
#endif
                    end if

                    this%X(n_alt_ii)(1, im, 1, ios, inn) = this%X(n_alt_ii)(1, im, 1, ios, inn) &
                      + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                    this%X(n_alt_ii)(1, im, 2, ios, inn) = this%X(n_alt_ii)(1, im, 2, ios, inn) &
                      + renorm_weight * phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                    this%X(n_alt_ii)(1, im, 3, ios, inn) = this%X(n_alt_ii)(1, im, 3, ios, inn) &
                      + renorm_weight * phi_dot_psi(1, im, ibatch, ios) * R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                    this%X(n_alt_ii)(1, im, 4, ios, inn) = this%X(n_alt_ii)(1, im, 4, ios, inn) &
                      + renorm_weight * phi_dot_psi(2, im, ibatch, ios) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))

                    do im2 = 1, os2%norbs

                      if (im == 1) then
                        this%X(n_alt_ii)(2, im2, 1, ios, inn) = this%X(n_alt_ii)(2, im2, 1, ios, inn) &
                          + renorm_weight * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im2, ibatch, ios2))
                        this%X(n_alt_ii)(2, im2, 2, ios, inn) = this%X(n_alt_ii)(2, im2, 2, ios, inn) &
                          + renorm_weight * phi_dot_psi(2, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(2, im2, ibatch, ios2))
                        this%X(n_alt_ii)(2, im2, 3, ios, inn) = this%X(n_alt_ii)(2, im2, 3, ios, inn) &
                          + renorm_weight * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(2, im2, ibatch, ios2))
                        this%X(n_alt_ii)(2, im2, 4, ios, inn) = this%X(n_alt_ii)(2, im2, 4, ios, inn) &
                          + renorm_weight * phi_dot_psi(2, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im2, ibatch, ios2))
                      end if


                      this%X(n_ij)(im, im2, 1, ios, inn) = this%X(n_ij)(im, im2, 1, ios, inn) &
                        + weight_phase*phi_dot_psi(1, im2, ibatch, ios2)*R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                      this%X(n_ij)(im, im2, 2, ios, inn) = this%X(n_ij)(im, im2, 2, ios, inn) &
                        + weight_phase*phi_dot_psi(2, im2, ibatch, ios2)*R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                      this%X(n_ij)(im, im2, 3, ios, inn) = this%X(n_ij)(im, im2, 3, ios, inn) &
                        + weight_phase*phi_dot_psi(1, im2, ibatch, ios2)*R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                      this%X(n_ij)(im, im2, 4, ios, inn) = this%X(n_ij)(im, im2, 4, ios, inn) &
                        + weight_phase*phi_dot_psi(2, im2, ibatch, ios2)*R_CONJ(phi_dot_psi(1, im, ibatch, ios))


                      this%X(n_alt_ij)(im, im2, 1, ios, inn) = this%X(n_alt_ij)(im, im2, 1, ios, inn) &
                        + renorm_weight_phase * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                      this%X(n_alt_ij)(im, im2, 2, ios, inn) = this%X(n_alt_ij)(im, im2, 2, ios, inn) &
                        + renorm_weight_phase * phi_dot_psi(2, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                      this%X(n_alt_ij)(im, im2, 3, ios, inn) = this%X(n_alt_ij)(im, im2, 3, ios, inn) &
                        + renorm_weight_phase * phi_dot_psi(1, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(2, im, ibatch, ios))
                      this%X(n_alt_ij)(im, im2, 4, ios, inn) = this%X(n_alt_ij)(im, im2, 4, ios, inn) &
                        + renorm_weight_phase * phi_dot_psi(2, im2, ibatch, ios2) * R_CONJ(phi_dot_psi(1, im, ibatch, ios))
                    end do !im2
                  end do !inn
                end if
              end if
            end if !Spinors
          end do !im
        end do !ios
      end do !ibatch
      if (present(phase)) then
        call epsib%end(copy=.false.)
        SAFE_DEALLOCATE_P(epsib)
      end if
      call st%group%psib(ib, ik)%do_unpack(copy = .false.)
    end do !ib
  end do !ik

  SAFE_DEALLOCATE_A(phi_dot_psi)
  SAFE_DEALLOCATE_A(psi)
  SAFE_DEALLOCATE_A(muliken_charge)

  if (st%parallel_in_states .or. st%d%kpt%parallel) then
    call comm_allreduce(st%st_kpt_mpi_grp, this%X(n))
    if (this%level == DFT_U_ACBN0) then
      call comm_allreduce(st%st_kpt_mpi_grp, this%X(n_alt))
      if(this%intersite .and. this%maxneighbors > 0) then
        call comm_allreduce(st%st_kpt_mpi_grp, this%X(n_ij))
        call comm_allreduce(st%st_kpt_mpi_grp, this%X(n_alt_ij))
        call comm_allreduce(st%st_kpt_mpi_grp, this%X(n_alt_ii))
      end if
    end if
  end if


  ! We compute the double counting mixing coefficient
  select case(this%double_couting)
  case(DFT_U_FLL)
    this%dc_alpha = M_ONE
  case(DFT_U_AMF)
    ASSERT(st%d%ispin /= SPINORS)
    this%dc_alpha = M_ZERO
  case(DFT_U_MIX) !See PRB 67, 153106 (2003), Eq.(5)
    ASSERT(st%d%ispin /= SPINORS)
    do ios = 1, this%norbsets
      denom = M_ZERO
      num = M_ZERO
      do ispin = 1, this%nspins
        nsigma = M_ZERO
        do im = 1, this%orbsets(ios)%norbs
          nsigma = nsigma + R_REAL(this%X(n)(im,im,ispin,ios))/st%smear%el_per_state
        end do
        denom = denom + nsigma*(M_ONE-nsigma/this%orbsets(ios)%norbs)

        ! We compute Tr(\delta\rho \delta\rho)
        do im = 1, this%orbsets(ios)%norbs
          do imp = 1, this%orbsets(ios)%norbs
            num = num + abs(this%X(n)(im, imp, ispin, ios))**2/st%smear%el_per_state
          end do
          num = num - nsigma/this%orbsets(ios)%norbs*real(this%X(n)(im, im, ispin, ios), real64)
        end do

      end do ! ispin
      this%dc_alpha(ios) = num / denom
      write(message(1),'(a,i3,a,f6.3)') "Double counting mixing for orb. set ", ios, " is ", this%dc_alpha(ios)
      if(this%dc_alpha(ios) < M_ZERO .or. this%dc_alpha(ios) > M_ONE) then
        call messages_warning(1)
      else if(debug%info) then
        call messages_info(1)
      end if

      this%dc_alpha(ios) = max(min(this%dc_alpha(ios), M_ONE), M_ZERO)
    end do ! ios
  end select

  ! We symmetrize the occupation matrices if needed
  if(this%symmetrize_occ_matrices) then
    call X(symmetrize_occ_matrices)(this, this%X(n))
    if (this%level == DFT_U_ACBN0) then
      call X(symmetrize_occ_matrices)(this, this%X(n_alt))
    end if
  end if

  if (this%level == DFT_U_ACBN0 .and. .not. this%freeze_u) then
    if (this%nspins > 1) then
      do ios = 1, this%norbsets
        if (this%orbsets(ios)%ndim  == 1) then
          call X(compute_ACBNO_U)(this, ios, namespace)
          call X(compute_ACBNO_V)(this, ios)
        else
          call compute_ACBNO_U_noncollinear(this, ios, namespace)
          call compute_ACBNO_V_noncollinear(this, ios)
        end if
      end do
    else
      call X(compute_ACBNO_U_restricted)(this)
      call X(compute_ACBNO_V_restricted)(this)
    end if
  end if


  call X(compute_dftu_energy)(this, lda_u_energy, st)
  call X(lda_u_update_potential)(this,st)

  POP_SUB(update_occ_matrices)
  call profiling_out(TOSTRING(X(DFTU_OCC_MATRICES)))
end subroutine X(update_occ_matrices)

! ---------------------------------------------------------
!>@brief Symmetrize the occupation matrices
!!
!! At the moment this is only done for the case of real spherical harmonics
!! using the fact that a rotated spherical harmonic \f$Y_m^l\f$ can be expressed in the basis of
!! the spherical harmonics of the l.
!!
!! The other cases (complex spherical harmonics, basis from state) are not symmetrized at the moment.
!! The intersite generalized occupation matrices are also not treated at the moment
subroutine X(symmetrize_occ_matrices)(this, occ)
  type(lda_u_t), intent(inout)    :: this
  R_TYPE,        intent(inout)    :: occ(:,:,:,:)

  R_TYPE, allocatable :: occ_tmp(:,:,:,:)
  integer :: ios, ispin, im, imp, im_sym, imp_sym
  integer :: norbs, ios_sym, iop

  PUSH_SUB(X(symmetrize_occ_matrices))

  SAFE_ALLOCATE(occ_tmp(1:this%maxnorbs, 1:this%maxnorbs, 1:this%nspins, 1:this%norbsets))
  occ_tmp = occ
  occ = M_ZERO

  do ios = 1, this%norbsets

    ! The complex spherical harmonics is not implemented here
    ! The basis from state case is more tricky as we do not know if the subspace is invariant under rotations
    if( this%orbsets(ios)%ndim > 1 .or. this%basisfromstates) then
      occ(:,:,:,ios) = occ_tmp(:,:,:,ios)
      cycle
    end if

    norbs = this%orbsets(ios)%norbs

    ! Loop over symmetries
    do iop = 1, this%nsym
      ios_sym = this%inv_map_symm(ios, iop)

      do ispin = 1, this%nspins
        do im = 1, norbs
          do imp = 1, norbs
            do im_sym = 1, norbs
              do imp_sym = 1, norbs
                occ(im, imp, ispin, ios) = occ(im, imp, ispin, ios) &
                  + this%symm_weight(im, im_sym, iop, ios) * this%symm_weight(imp, imp_sym, iop, ios) &
                  * occ_tmp(im_sym, imp_sym, ispin, ios_sym) / this%nsym
              end do !imp_sym
            end do !im_sym
          end do !imp
        end do !im
      end do !ispin
    end do !iop
  end do !ios

  SAFE_DEALLOCATE_A(occ_tmp)

  POP_SUB(X(symmetrize_occ_matrices))
end subroutine X(symmetrize_occ_matrices)

! ---------------------------------------------------------
!>@brief This routine computes the value of the double counting term in the DFT+U energy
subroutine X(compute_dftu_energy)(this, energy, st)
  type(lda_u_t), intent(inout)    :: this
  real(real64), intent(inout)            :: energy
  type(states_elec_t), intent(in) :: st

  integer :: ios, imp, im, ispin, inn
  type(orbitalset_t), pointer :: os
  real(real64) :: nsigma

  PUSH_SUB(compute_dftu_energy)

  energy = M_ZERO

  ! See for instance PRB 67, 153106 (2003), Eq.(3)
  do ios = 1, this%norbsets
    do ispin = 1, this%nspins
      do im = 1, this%orbsets(ios)%norbs
        do imp = 1, this%orbsets(ios)%norbs
          energy = energy - M_HALF*this%orbsets(ios)%Ueff &
            *abs(this%X(n)(im, imp, ispin, ios))**2/st%smear%el_per_state
        end do
        if (ispin <= this%spin_channels) then
          energy = energy + M_HALF*this%orbsets(ios)%Ueff*real(this%X(n)(im, im, ispin, ios), real64)
        end if
      end do
    end do
  end do

  ! This is independent of the double-counting form
  if (this%intersite) then
    do ios = 1, this%norbsets
      os => this%orbsets(ios)
      do inn = 1, os%nneighbors
        do ispin = 1, this%nspins
          do im = 1, os%norbs
            do imp = 1, this%orbsets(os%map_os(inn))%norbs
              energy = energy - M_HALF*os%V_ij(inn,0)/st%smear%el_per_state &
                *abs(this%X(n_ij)(im, imp, ispin, ios, inn))**2
            end do
          end do
        end do
      end do
    end do
  end if

  ! We already computed the contribution from Tr(\rho\rho) - N_sigma
  ! We then add -N_sigma (1-\alpha)(n_sigma-1)
  ! See PRB 67, 153106 (2003), Eq.(4)
  do ios = 1, this%norbsets
    if(this%dc_alpha(ios) < M_ONE) then
      ASSERT(st%d%ispin /= SPINORS)
      do ispin = 1, this%nspins
        nsigma = M_ZERO
        do im = 1, this%orbsets(ios)%norbs
          nsigma = nsigma + R_REAL(this%X(n)(im,im,ispin,ios))/st%smear%el_per_state
        end do

        do im = 1, this%orbsets(ios)%norbs
          energy = energy + M_HALF*this%orbsets(ios)%Ueff*nsigma*(M_ONE-this%dc_alpha(ios)) &
            *(nsigma/this%orbsets(ios)%norbs-M_ONE)
        end do
      end do
    end if
  end do

  POP_SUB(compute_dftu_energy)
end subroutine X(compute_dftu_energy)


! ---------------------------------------------------------
!> This routine computes the potential that, once multiplied
!> by the projector Pmm' and summed over m and m' for all the atoms
!> gives the full Hubbard potential
! ---------------------------------------------------------
subroutine X(lda_u_update_potential)(this, st)
  type(lda_u_t), intent(inout)    :: this
  type(states_elec_t), intent(in) :: st

  integer :: ios, im, ispin, norbs
  real(real64) :: nsigma

  call profiling_in(TOSTRING(X(DFTU_POTENTIAL)))

  PUSH_SUB(lda_u_update_potential)

  this%X(V) = M_ZERO

  do ios = this%orbs_dist%start, this%orbs_dist%end
    norbs = this%orbsets(ios)%norbs
    do ispin = 1, this%nspins
      do im = 1, norbs
        this%X(V)(1:norbs,im,ispin,ios) = - this%orbsets(ios)%Ueff*this%X(n)(1:norbs,im,ispin,ios)/st%smear%el_per_state
        ! Only the diagonal part in spin space (for spinors)
        if (ispin <= this%spin_channels) then
          this%X(V)(im,im,ispin,ios) = this%X(V)(im,im,ispin,ios) + M_HALF*this%orbsets(ios)%Ueff
        end if

        if (this%orbsets(ios)%alpha > 1.0e-6_real64) then
          this%X(V)(im,im,ispin,ios) = this%X(V)(im,im,ispin,ios) + this%orbsets(ios)%alpha
        end if
      end do
    end do

    !See for instance PRB 67, 153106 (2003), Eq.(4)
    if(this%dc_alpha(ios) < M_ONE) then
      ASSERT(st%d%ispin /= SPINORS)
      do ispin = 1, this%nspins
        nsigma = M_ZERO
        do im = 1, norbs
          nsigma = nsigma + R_REAL(this%X(n)(im,im,ispin,ios))/st%smear%el_per_state
        end do
        do im = 1, norbs
          this%X(V)(im,im,ispin,ios) = this%X(V)(im,im,ispin,ios) + this%orbsets(ios)%Ueff &
            *(M_ONE-this%dc_alpha(ios))*(nsigma/norbs - M_HALF)
        end do
      end do
    end if
  end do


  if (this%orbs_dist%parallel) then
    call comm_allreduce(this%orbs_dist%mpi_grp, this%X(V))
  end if

  POP_SUB(lda_u_update_potential)
  call profiling_out(TOSTRING(X(DFTU_POTENTIAL)))
end subroutine X(lda_u_update_potential)

! ---------------------------------------------------------
!> This routine computes the effective U following the expression
!> given in Agapito et al., Phys. Rev. X 5, 011006 (2015)
! ---------------------------------------------------------
subroutine X(compute_ACBNO_U)(this, ios, namespace)
  type(lda_u_t),     intent(inout) :: this
  integer,           intent(in)    :: ios
  type(namespace_t), intent(in)    :: namespace

  integer :: im, imp, impp, imppp, ispin1, ispin2, norbs
  real(real64)   :: numU, numJ, denomU, denomJ, tmpU, tmpJ

  PUSH_SUB(compute_ACBNO_U)

  ASSERT(this%orbsets(ios)%ndim == 1)

  norbs = this%orbsets(ios)%norbs
  numU = M_ZERO
  numJ = M_ZERO
  denomU = M_ZERO
  denomJ = M_ZERO

  if (norbs > 1) then

    do im = 1, norbs
      do imp = 1,norbs
        do impp = 1, norbs
          do imppp = 1, norbs
            ! We first compute the terms
            ! sum_{alpha,beta} P^alpha_{mmp}P^beta_{mpp,mppp}
            ! sum_{alpha} P^alpha_{mmp}P^alpha_{mpp,mppp}
            tmpU = M_ZERO
            tmpJ = M_ZERO

            do ispin1 = 1, this%spin_channels
              do ispin2 = 1, this%spin_channels
                tmpU = tmpU + R_REAL(this%X(n_alt)(im,imp,ispin1,ios)*this%X(n_alt)(impp,imppp,ispin2,ios))
              end do
              tmpJ = tmpJ + R_REAL(this%X(n_alt)(im,imp,ispin1,ios)*this%X(n_alt)(impp,imppp,ispin1,ios))
            end do
            if (this%nspins>this%spin_channels) then !Spinors
              tmpJ = tmpJ + R_REAL(this%X(n_alt)(im,imp,3,ios)*this%X(n_alt)(impp,imppp,4,ios)) &
                +R_REAL(this%X(n_alt)(im,imp,4,ios)*this%X(n_alt)(impp,imppp,3,ios))
            end if
            ! These are the numerator of the ACBN0 U and J
            numU = numU + tmpU*this%coulomb(im,imp,impp,imppp,ios)
            numJ = numJ + tmpJ*this%coulomb(im,imppp,impp,imp,ios)
          end do
        end do

        ! We compute the term
        ! sum_{alpha} sum_{m,mp/=m} N^alpha_{m}N^alpha_{mp}
        tmpJ = M_ZERO
        tmpU = M_ZERO
        if (imp/=im) then
          do ispin1 = 1, this%spin_channels
            tmpJ = tmpJ + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin1,ios))
            tmpU = tmpU + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin1,ios))
          end do
          if (this%nspins>this%spin_channels) then !Spinors
            tmpJ = tmpJ + R_REAL(this%X(n)(im,im,3,ios)*this%X(n)(imp,imp,4,ios)) &
              +R_REAL(this%X(n)(im,im,4,ios)*this%X(n)(imp,imp,3,ios))
          end if
        end if

        ! We compute the term
        ! sum_{alpha,beta} sum_{m,mp} N^alpha_{m}N^beta_{mp}
        do ispin1 = 1, this%spin_channels
          do ispin2 = 1, this%spin_channels
            if (ispin1 /= ispin2) then
              tmpU = tmpU + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin2,ios))
            end if
          end do
        end do

        if (this%nspins>this%spin_channels) then !Spinors
          if (im == imp) then
            tmpU = tmpU -(R_REAL(this%X(n)(im,im,3,ios)*this%X(n)(im,im,4,ios)) &
              +R_REAL(this%X(n)(im,im,4,ios)*this%X(n)(im,im,3,ios)))
          end if
        end if

        if (this%rot_inv) then
          !Rotationally invariance term
          !sum_{alpha} sum_{m,mp/=m} n^alpha_{mmp}n^alpha_{mpm}
          if (imp/=im) then
            do ispin1 = 1, this%spin_channels
              tmpJ = tmpJ + R_REAL(this%X(n)(im,imp,ispin1,ios)*this%X(n)(imp,im,ispin1,ios))
              tmpU = tmpU + R_REAL(this%X(n)(im,imp,ispin1,ios)*this%X(n)(imp,im,ispin1,ios))
            end do
            ASSERT(this%nspins == this%spin_channels) !Spinors not yet implemented
          end if
        end if

        denomJ = denomJ + tmpJ
        denomU = denomU + tmpU

      end do
    end do

    this%orbsets(ios)%Ueff = numU/(max(denomU,1e-6_real64)) - numJ/(max(denomJ, 1e-6_real64))
    this%orbsets(ios)%Ubar = numU/(max(denomU,1e-6_real64))
    this%orbsets(ios)%Jbar = numJ/(max(denomJ, 1e-6_real64))

  else !In the case of s orbitals, the expression is different
    ! sum_{alpha/=beta} P^alpha_{mmp}P^beta_{mpp,mppp}
    ! sum_{alpha,beta} sum_{m,mp} N^alpha_{m}N^beta_{mp}
    numU = M_ZERO
    denomU = M_ZERO
    do ispin1 = 1, this%spin_channels
      do ispin2 = 1, this%spin_channels
        if (ispin1 /= ispin2) then
          numU = numU + R_REAL(this%X(n_alt)(1,1,ispin1,ios))*R_REAL(this%X(n_alt)(1,1,ispin2,ios))
          denomU = denomU + R_REAL(this%X(n)(1,1,ispin1,ios))*R_REAL(this%X(n)(1,1,ispin2,ios))
        end if
      end do
    end do

    if (this%nspins>this%spin_channels) then !Spinors
      denomU = denomU - (R_REAL(this%X(n)(1,1,3,ios)*this%X(n)(1,1,4,ios)) &
        +R_REAL(this%X(n)(1,1,4,ios)*this%X(n)(1,1,3,ios)))
    end if

    ! We have to be careful in the case of hydrogen atom for instance
    if (abs(denomU)> 1.0e-3_real64) then
      this%orbsets(ios)%Ubar = (numU/denomU)*R_REAL(this%coulomb(1,1,1,1,ios))
    else
      if (abs(numU-denomU) < 1.0e-3_real64) then
        this%orbsets(ios)%Ubar = R_REAL(this%coulomb(1,1,1,1,ios))
      else
        this%orbsets(ios)%Ubar = (numU/denomU)
        write(message(1),'(a,a)')' Small denominator value for the s orbital ', this%orbsets(ios)%Ubar
        write(message(2),'(a,a)')' to be multiplied by ',  this%coulomb(1,1,1,1,ios)
        call messages_warning(2, namespace=namespace)
        this%orbsets(ios)%Ubar = this%orbsets(ios)%Ubar*this%coulomb(1,1,1,1,ios)
      end if
    end if

    this%orbsets(ios)%Jbar = 0
    this%orbsets(ios)%Ueff = this%orbsets(ios)%Ubar
  end if

  POP_SUB(compute_ACBNO_U)
end subroutine X(compute_ACBNO_U)


! ---------------------------------------------------------
!> This routine computes the effective Uin the spin-unpolarised case
! ---------------------------------------------------------
subroutine X(compute_ACBNO_U_restricted)(this)
  type(lda_u_t), intent(inout)    :: this

  integer :: ios, im, imp, impp, imppp, norbs
  real(real64)   :: numU, numJ, denomU, denomJ

  PUSH_SUB(compute_ACBNO_U_restricted)

  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    numU = M_ZERO
    numJ = M_ZERO
    denomU = M_ZERO
    denomJ = M_ZERO

    if (norbs > 1) then

      do im = 1, norbs
        do imp = 1,norbs
          do impp = 1, norbs
            do imppp = 1, norbs
              numU = numU + R_REAL(this%X(n_alt)(im,imp,1,ios)*this%X(n_alt)(impp,imppp,1,ios)) &
                *this%coulomb(im,imp,impp,imppp,ios)
              numJ = numJ + R_REAL(this%X(n_alt)(im,imp,1,ios)*this%X(n_alt)(impp,imppp,1,ios)) &
                *this%coulomb(im,imppp,impp,imp,ios)
            end do
          end do
          ! We compute the term
          ! sum_{m,mp/=m} N_{m}N_{mp}
          if (imp/=im) then
            denomJ = denomJ + R_REAL(this%X(n)(im,im,1,ios))*R_REAL(this%X(n)(imp,imp,1,ios))
            denomU = denomU + R_REAL(this%X(n)(im,im,1,ios))*R_REAL(this%X(n)(imp,imp,1,ios))
          end if
          ! We compute the term
          ! sum_{m,mp} N_{m}N_{mp}
          denomU = denomU + R_REAL(this%X(n)(im,im,1,ios))*R_REAL(this%X(n)(imp,imp,1,ios))

          if (this%rot_inv) then
            !Rotationally invariance term
            !sum_{m,mp/=m} n^alpha_{mmp}n^alpha_{mpm}
            if (imp/=im) then
              denomJ = denomJ + R_REAL(this%X(n)(im,imp,1,ios)*this%X(n)(imp,im,1,ios))
              denomU = denomU + R_REAL(this%X(n)(im,imp,1,ios)*this%X(n)(imp,im,1,ios))
            end if
          end if
        end do
      end do

      this%orbsets(ios)%Ueff = M_TWO*numU/denomU - numJ/denomJ
      this%orbsets(ios)%Ubar = M_TWO*numU/denomU
      this%orbsets(ios)%Jbar = numJ/denomJ

    else !In the case of s orbitals, the expression is different
      ! P_{mmp}P_{mpp,mppp}(m,mp|mpp,mppp)
      numU = R_REAL(this%X(n_alt)(1,1,1,ios))**2*this%coulomb(1,1,1,1,ios)

      ! We compute the term
      ! sum_{alpha,beta} sum_{m,mp} N^alpha_{m}N^beta_{mp}
      denomU = R_REAL(this%X(n)(1,1,1,ios))**2

      this%orbsets(ios)%Ueff = numU/denomU
      this%orbsets(ios)%Ubar = numU/denomU
      this%orbsets(ios)%Jbar = 0
    end if
  end do

  POP_SUB(compute_ACBNO_U_restricted)
end subroutine X(compute_ACBNO_U_restricted)

! ---------------------------------------------------------
!> This routine computes the effective V in the spin-polarized case
! ---------------------------------------------------------
subroutine X(compute_ACBNO_V)(this, ios)
  type(lda_u_t), intent(inout)    :: this
  integer,       intent(in)       :: ios

  integer :: ios2, im, imp
  integer :: inn, norbs, norbs2
  integer :: ispin1, ispin2
  real(real64)   :: numV, denomV

  if (.not. this%intersite) return

  PUSH_SUB(compute_ACBNO_V)

  norbs = this%orbsets(ios)%norbs
  do inn = 1, this%orbsets(ios)%nneighbors
    numV = M_ZERO
    denomV = M_ZERO

    ios2 = this%orbsets(ios)%map_os(inn)
    norbs2 = this%orbsets(ios2)%norbs

    do im = 1, norbs
      do imp= 1, norbs2
        do ispin1 = 1, this%spin_channels
          do ispin2 = 1, this%spin_channels
            numV = numV + R_REAL(this%X(n_alt_ii)(1,im,ispin1,ios,inn))*R_REAL(this%X(n_alt_ii)(2,imp,ispin2,ios,inn))   &
              *this%orbsets(ios)%coulomb_IIJJ(im,im,imp,imp,inn)
            if (ispin1 == ispin2) then
              numV = numV - R_REAL(this%X(n_alt_ij)(im,imp,ispin1,ios,inn)*R_CONJ(this%X(n_alt_ij)(im,imp,ispin1,ios,inn)))&
                *this%orbsets(ios)%coulomb_IIJJ(im,im,imp,imp,inn)
            end if
          end do
        end do
        if (this%nspins>this%spin_channels) then !Spinors
          numV = numV -(R_REAL(this%X(n_alt_ij)(im,imp,3,ios,inn)*R_CONJ(this%X(n_alt_ij)(im,imp,3,ios,inn))) &
            +R_REAL(this%X(n_alt_ij)(im,imp,4,ios,inn)*R_CONJ(this%X(n_alt_ij)(im,imp,4,ios,inn)))) &
            *this%orbsets(ios)%coulomb_IIJJ(im,im,imp,imp,inn)
        end if
      end do !imp

      do imp = 1,norbs2
        ! We compute the term
        ! sum_{m,mp} ( 2*N_{m}N_{mp} - n_{mmp}n_{mpm})
        do ispin1 = 1, this%spin_channels
          do ispin2 = 1, this%spin_channels
            denomV = denomV + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin2,ios2))
            if (ispin1 == ispin2) denomV = denomV - abs(this%X(n_ij)(im,imp,ispin1,ios,inn))**2
          end do
        end do
        if (this%nspins>this%spin_channels) then !Spinors
          denomV = denomV - abs(this%X(n_ij)(im,imp,3,ios,inn))**2 - abs(this%X(n_ij)(im,imp,4,ios,inn))**2
        end if
      end do !imp
    end do !im


    this%orbsets(ios)%V_ij(inn,0) = numV/denomV*M_HALF
  end do !inn

  POP_SUB(compute_ACBNO_V)
end subroutine X(compute_ACBNO_V)



! ---------------------------------------------------------
!> This routine computes the effective V in the spin-unpolarised case
! ---------------------------------------------------------
subroutine X(compute_ACBNO_V_restricted)(this)
  type(lda_u_t), intent(inout)    :: this

  integer :: ios, ios2, im, imp
  integer :: inn, norbs, norbs2
  real(real64)   :: numV, denomV

  if (.not. this%intersite) return

  PUSH_SUB(compute_ACBNO_V_restricted)

  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    do inn = 1, this%orbsets(ios)%nneighbors
      numV = M_ZERO
      denomV = M_ZERO

      ios2 = this%orbsets(ios)%map_os(inn)
      norbs2 = this%orbsets(ios2)%norbs

      do im = 1, norbs
        do imp = 1, norbs2
          numV = numV + (M_TWO*R_REAL(this%X(n_alt_ii)(1,im,1,ios,inn)*this%X(n_alt_ii)(2,imp,1,ios,inn))   &
            - R_REAL(this%X(n_alt_ij)(im,imp,1,ios,inn)*R_CONJ(this%X(n_alt_ij)(im,imp,1,ios,inn))))&
            *this%orbsets(ios)%coulomb_IIJJ(im,im,imp,imp,inn)
        end do

        do imp = 1, norbs2
          ! We compute the term
          ! sum_{m,mp} ( 2*N_{m}N_{mp} - n_{mmp}n_{mpm})
          denomV = denomV + M_TWO*R_REAL(this%X(n)(im,im,1,ios))*R_REAL(this%X(n)(imp,imp,1,ios2)) &
            - abs(this%X(n_ij)(im,imp,1,ios,inn))**2
        end do
      end do

      this%orbsets(ios)%V_ij(inn,0) = numV/denomV * M_HALF
    end do !inn
  end do !ios

  POP_SUB(compute_ACBNO_V_restricted)
end subroutine X(compute_ACBNO_V_restricted)

! ---------------------------------------------------------
!> This routine computes the Kanamori U, Up, and J
! ---------------------------------------------------------
subroutine X(compute_ACBNO_U_kanamori)(this, kanamori)
  type(lda_u_t), intent(in)       :: this
  real(real64),  intent(out)      :: kanamori(:,:)

  integer :: ios, im, imp, impp, imppp, norbs
  integer :: ispin1, ispin2
  real(real64)   :: numU, numUp, numJ, denomU, denomUp, denomJ
  real(real64)   :: tmpU, tmpUp, tmpJ

  PUSH_SUB(compute_ACBNO_U_kanamori)

  ASSERT(this%level == DFT_U_ACBN0)

  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    numU = M_ZERO
    denomU = M_ZERO
    numUp = M_ZERO
    denomUp = M_ZERO
    numJ = M_ZERO
    denomJ = M_ZERO

    if (norbs > 1) then

      do im = 1, norbs
        do imp = 1,norbs
          do impp = 1, norbs
            do imppp = 1, norbs
              tmpU = M_ZERO
              tmpUp = M_ZERO
              tmpJ = M_ZERO

              do ispin1 = 1, this%spin_channels
                do ispin2 = 1, this%spin_channels
                  tmpUp = tmpUp + R_REAL(this%X(n_alt)(im,imp,ispin1,ios)*this%X(n_alt)(impp,imppp,ispin2,ios))
                  if (ispin1 /= ispin2) then
                    tmpU = tmpU + R_REAL(this%X(n_alt)(im,imp,ispin1,ios)*this%X(n_alt)(impp,imppp,ispin2,ios))
                  end if
                end do
                tmpJ = tmpJ + R_REAL(this%X(n_alt)(im,imp,ispin1,ios)*this%X(n_alt)(impp,imppp,ispin1,ios))
              end do

              if (im == imp .and. impp == imppp .and. im == impp) then
                numU = numU + tmpU*this%coulomb(im,imp,impp,imppp,ios)
              else
                numUp = numUp + tmpUp*this%coulomb(im,imp,impp,imppp,ios)
                numJ = numJ + tmpJ*this%coulomb(im,imppp,impp,imp,ios)
              end if
            end do
          end do

          tmpU = M_ZERO
          tmpUp = M_ZERO
          tmpJ = M_ZERO
          if (imp/=im) then
            do ispin1 = 1, this%spin_channels
              tmpUp = tmpUp + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin1,ios))
              tmpJ = tmpJ   + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin1,ios))
            end do
          end if

          do ispin1 = 1, this%spin_channels
            do ispin2 = 1, this%spin_channels
              if (ispin1 /= ispin2) then
                if (im /= imp) then
                  tmpUp = tmpUp + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin2,ios))
                else
                  tmpU = tmpU + R_REAL(this%X(n)(im,im,ispin1,ios))*R_REAL(this%X(n)(imp,imp,ispin2,ios))
                end if
              end if
            end do
          end do

          denomU = denomU + tmpU
          denomUp = denomUp + tmpUp
          denomJ = denomJ + tmpJ

        end do
      end do

      kanamori(1,ios) = numU/denomU
      kanamori(2,ios) = numUp/denomUp
      kanamori(3,ios) = numJ/denomJ

    else !In the case of s orbitals, the expression is different
      kanamori(1,ios) = this%orbsets(ios)%Ubar
      kanamori(2,ios) = M_ZERO
      kanamori(3,ios) = M_ZERO
    end if


  end do

  POP_SUB(compute_ACBNO_U_kanamori)
end subroutine X(compute_ACBNO_U_kanamori)

! ---------------------------------------------------------
!> This routine computes the Kanamori U, Up, and J
! ---------------------------------------------------------
subroutine X(compute_ACBNO_U_kanamori_restricted)(this, kanamori)
  type(lda_u_t), intent(in)       :: this
  real(real64),  intent(out)      :: kanamori(3)

  integer :: ios, im, imp, impp, imppp, norbs
  real(real64)   :: numU, numUp, numJ, denomU, denomUp, denomJ
  real(real64)   :: tmpU, tmpUp, tmpJ

  PUSH_SUB(compute_ACBNO_U_kanamori_restricted)

  ASSERT(this%nspins == 1)

  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    numU = M_ZERO
    denomU = M_ZERO
    numUp = M_ZERO
    denomUp = M_ZERO
    numJ = M_ZERO
    denomJ = M_ZERO

    if (norbs > 1) then

      do im = 1, norbs
        do imp = 1,norbs
          do impp = 1, norbs
            do imppp = 1, norbs
              tmpU = M_ZERO
              tmpUp = M_ZERO
              tmpJ = M_ZERO

              tmpUp = tmpUp + M_TWO*R_REAL(this%X(n_alt)(im,imp,1,ios)*this%X(n_alt)(impp,imppp,1,ios))
              tmpU = tmpU + R_REAL(this%X(n_alt)(im,imp,1,ios)*this%X(n_alt)(impp,imppp,1,ios))
              tmpJ = tmpJ + R_REAL(this%X(n_alt)(im,imp,1,ios)*this%X(n_alt)(impp,imppp,1,ios))

              ! These are the numerator of the ACBN0 U and J
              if (im == imp .and. impp == imppp .and. im == impp) then
                numU = numU + tmpU*this%coulomb(im,imp,impp,imppp,ios)
              else
                numUp = numUp + tmpUp*this%coulomb(im,imp,impp,imppp,ios)
                numJ = numJ + tmpU*this%coulomb(im,imppp,impp,imp,ios)
              end if
            end do
          end do

          if (im /= imp) then
            denomUp = denomUp + M_TWO*R_REAL(this%X(n)(im,im,1,ios)*this%X(n)(imp,imp,1,ios))
            denomJ = denomJ + R_REAL(this%X(n)(im,im,1,ios))*R_REAL(this%X(n)(imp,imp,1,ios))
          else
            denomU = denomU + R_REAL(this%X(n)(im,im,1,ios)*this%X(n)(imp,imp,1,ios))
          end if

        end do
      end do

      kanamori(1) = numU/denomU
      kanamori(2) = numUp/denomUp
      kanamori(3) = numJ/denomJ


    else !In the case of s orbitals, the expression is different
      kanamori(1) = this%orbsets(ios)%Ubar
      kanamori(2) = M_ZERO
      kanamori(3) = M_ZERO
    end if


  end do

  POP_SUB(compute_ACBNO_U_kanamori_restricted)
end subroutine X(compute_ACBNO_U_kanamori_restricted)


! ---------------------------------------------------------
! TODO: Merge this with the two_body routine in states/elec_matrix_elements_inc.F90
subroutine X(compute_coulomb_integrals) (this, namespace, space, gr, psolver)
  type(lda_u_t),       intent(inout)  :: this
  type(namespace_t),   intent(in)     :: namespace
  class(space_t),      intent(in)     :: space
  type(grid_t),        intent(in)     :: gr
  type(poisson_t),     intent(in)     :: psolver

  integer :: ist, jst, kst, lst, ijst, klst
  integer :: norbs, np_sphere, ios, ip
  integer :: idone, ntodo
  real(real64), allocatable :: tmp(:), vv(:), nn(:)
  type(orbitalset_t), pointer :: os

  call profiling_in(TOSTRING(X(DFTU_COULOMB_INTEGRALS)))

  PUSH_SUB(X(compute_coulomb_integrals))

  SAFE_ALLOCATE(nn(1:this%max_np))
  SAFE_ALLOCATE(vv(1:this%max_np))
  SAFE_ALLOCATE(tmp(1:this%max_np))

  this%coulomb(1:this%maxnorbs,1:this%maxnorbs,1:this%maxnorbs,1:this%maxnorbs,1:this%norbsets) = M_ZERO

  !Lets counts the number of orbital to treat, to display a progress bar
  ntodo = 0
  do ios = this%orbs_dist%start, this%orbs_dist%end
    norbs = this%orbsets(ios)%norbs
    ntodo= ntodo + ((norbs+1)*norbs/2)*((norbs+1)*norbs/2+1)/2
  end do
  idone = 0
  if (mpi_world%rank == 0) call loct_progress_bar(-1, ntodo)


  do ios = this%orbs_dist%start, this%orbs_dist%end
    os => this%orbsets(ios)
    norbs = os%norbs
    np_sphere = os%sphere%np

    select case (this%sm_poisson)
    case (SM_POISSON_DIRECT)
      call submesh_build_global(os%sphere, space)
      call poisson_init_sm(os%poisson, namespace, space, psolver, gr%der, os%sphere, method = POISSON_DIRECT_SUM)
    case (SM_POISSON_ISF)
      call poisson_init_sm(os%poisson, namespace, space, psolver, gr%der, os%sphere, method = POISSON_ISF)
    case (SM_POISSON_PSOLVER)
      call poisson_init_sm(os%poisson, namespace, space, psolver, gr%der, os%sphere, method = POISSON_PSOLVER)
    case (SM_POISSON_FFT)
      call poisson_init_sm(os%poisson, namespace, space, psolver, gr%der, os%sphere, method = POISSON_FFT)
    end select

    ijst=0
    do ist = 1, norbs

      do jst = 1, norbs
        if (jst > ist) cycle
        ijst=ijst+1

        !$omp parallel do
        do ip = 1, np_sphere
          nn(ip) = real(os%X(orb)(ip, 1, ist), real64)*real(os%X(orb)(ip, 1, jst), real64)
        end do
        !$omp end parallel do

        !Here it is important to use a non-periodic poisson solver, e.g. the direct solver
        call dpoisson_solve_sm(os%poisson, namespace, os%sphere, vv(1:np_sphere), nn(1:np_sphere))

        klst=0
        do kst = 1, norbs
          do lst = 1, norbs
            if (lst > kst) cycle
            klst=klst+1
            if (klst > ijst) cycle

            !$omp parallel do
            do ip = 1, np_sphere
              tmp(ip) = vv(ip)*real(os%X(orb)(ip,1,kst), real64)*real(os%X(orb)(ip,1,lst), real64)
            end do
            !$omp end parallel do

            this%coulomb(ist,jst,kst,lst,ios) = dsm_integrate(gr, os%sphere, tmp(1:np_sphere), reduce = .false.)

            this%coulomb(kst,lst,ist,jst,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(jst,ist,lst,kst,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(lst,kst,jst,ist,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(jst,ist,kst,lst,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(lst,kst,ist,jst,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(ist,jst,lst,kst,ios) = this%coulomb(ist,jst,kst,lst,ios)
            this%coulomb(kst,lst,jst,ist,ios) = this%coulomb(ist,jst,kst,lst,ios)

            !Update the progress bar
            idone = idone + 1
            if (mpi_world%rank == 0) call loct_progress_bar(idone, ntodo)
          end do !lst
        end do !kst
      end do !jst
    end do !ist
    call poisson_end(os%poisson)
    call submesh_end_cube_map(os%sphere)
    if(this%sm_poisson == SM_POISSON_DIRECT) call submesh_end_global(os%sphere)
  end do !iorb

  call gr%allreduce(this%coulomb)

  if (this%orbs_dist%parallel) then
    call comm_allreduce(this%orbs_dist%mpi_grp, this%coulomb)
  end if

  SAFE_DEALLOCATE_A(nn)
  SAFE_DEALLOCATE_A(vv)
  SAFE_DEALLOCATE_A(tmp)

  POP_SUB(X(compute_coulomb_integrals))
  call profiling_out(TOSTRING(X(DFTU_COULOMB_INTEGRALS)))
end subroutine X(compute_coulomb_integrals)

 ! ---------------------------------------------------------
subroutine X(lda_u_commute_r_single) (this, mesh, space, d, namespace, ist, ik, psi, gpsi, has_phase)
  type(lda_u_t),                 intent(in) :: this
  class(mesh_t),                 intent(in) :: mesh
  class(space_t),                intent(in) :: space
  type(states_elec_dim_t),       intent(in) :: d
  type(namespace_t),             intent(in) :: namespace
  integer,                       intent(in) :: ist, ik
  R_TYPE, contiguous, target, intent(inout) :: psi(:,:)
  R_TYPE, contiguous, target, intent(inout) :: gpsi(:,:,:) !< (1:gr%np, 1:gr%der%dim, 1:st%d%dim)
  logical,                       intent(in) :: has_phase !True if the wavefunction has an associated phase

  integer :: idir
  type(wfs_elec_t) :: psib
  class(wfs_elec_t), allocatable :: gpsib(:)
  R_TYPE, allocatable :: gpsi_tmp(:,:,:)

  if (this%level == DFT_U_NONE) return

  PUSH_SUB(X(lda_u_commute_r_single))

  SAFE_ALLOCATE_TYPE_ARRAY(wfs_elec_t, gpsib, (1:space%dim))

  call wfs_elec_init(psib, d%dim, ist, ist, psi, ik)
  psib%has_phase = has_phase
  SAFE_ALLOCATE(gpsi_tmp(1:mesh%np, 1:d%dim, 1:space%dim))
  do idir = 1, space%dim
    gpsi_tmp(1:mesh%np, 1:d%dim, idir) = gpsi(1:mesh%np, idir, 1:d%dim)
    call wfs_elec_init(gpsib(idir), d%dim, ist, ist, gpsi_tmp(:,:,idir), ik)
    gpsib(idir)%has_phase = has_phase
  end do

  call X(lda_u_commute_r)(this, mesh, space, d, namespace, psib, gpsib)

  call psib%end(copy = .false.)
  do idir = 1, space%dim
    call gpsib(idir)%end()
  end do
  SAFE_DEALLOCATE_A(gpsib)

  do idir = 1, space%dim
    gpsi(1:mesh%np, idir, 1:d%dim) = gpsi_tmp(1:mesh%np, 1:d%dim, idir)
  end do
  SAFE_DEALLOCATE_A(gpsi_tmp)

  POP_SUB(X(lda_u_commute_r_single))
end subroutine X(lda_u_commute_r_single)


! ---------------------------------------------------------
!> This routine computes [r,V_lda+u] .
! ---------------------------------------------------------
subroutine X(lda_u_commute_r)(this, mesh, space, d, namespace, psib, gpsib)
  type(lda_u_t),           intent(in) :: this
  class(mesh_t),           intent(in) :: mesh
  class(space_t),          intent(in) :: space
  type(states_elec_dim_t), intent(in) :: d
  type(namespace_t),       intent(in) :: namespace
  type(wfs_elec_t),        intent(in) :: psib
  class(wfs_elec_t),    intent(inout) :: gpsib(:)

  integer :: ios, idim, idir, im, is
  integer :: idim_orb
  R_TYPE, allocatable :: phi_dot_psi(:,:,:,:), weight(:,:,:), epsi(:,:)
  type(orbitalset_t), pointer  :: os
  logical :: use_submesh
  type(wfs_elec_t) :: rpsib
  real(real64), allocatable :: x_submesh(:)

  if (this%level == DFT_U_NONE) return

  PUSH_SUB(lda_u_commute_r)

  call profiling_in(TOSTRING(X(DFTU_COMMUTE_R)))

  if (this%double_couting /= DFT_U_FLL) then
    call messages_not_implemented("AMF double couting and commutator [r,V_u]", namespace=namespace)
  end if

  use_submesh = this%basis%use_submesh
  ! Because of possible phase corrections at the border, the array X(orb) is always stored
  ! on the submesh for complex wavefunctions
  ! This does only apply to X(orb). eorb_mesh/eorb_submesh are
  ! still stored according to the user choice given by basis%submesh.
  ! Hence, only if we do not have phases but have a complex wavefunctions we will access X(orb)
  ! always on the submesh
#ifdef R_TCOMPLEX
  if (.not. psib%has_phase) use_submesh = .true.
#endif

  SAFE_ALLOCATE(phi_dot_psi(1:d%dim,1:this%maxnorbs, 1:psib%nst, 1:this%norbsets))
  SAFE_ALLOCATE(weight(1:this%maxnorbs, 1:psib%nst_linear, 1:this%norbsets))

  do ios = 1, this%norbsets
    ! We have to compute
    ! hpsi> += r sum_m |phi m> sum_mp Vmmp <phi mp | psi >
    !
    ! We first compute <phi m | psi> for all orbitals of the atom
    !
    os => this%orbsets(ios)
    !
    call X(orbitalset_get_coeff_batch)(os, d%dim, psib, phi_dot_psi(:,:,:,ios))
  end do
  !
  weight(:,:,:) = M_ZERO
  !
  do ios = 1, this%norbsets
    ! We now compute weight = sum_mp Vmmp <phi mp | psi >
    os => this%orbsets(ios)
    !
    call X(dft_u_get_weight_projectors)(this, os, ios, d%ispin, psib, phi_dot_psi, &
      weight(1:os%norbs, 1:psib%nst_linear, ios))
  end do !ios

  SAFE_ALLOCATE(epsi(1:mesh%np, 1:d%dim))

  do ios = 1, this%norbsets
    os => this%orbsets(ios)

    SAFE_ALLOCATE(x_submesh(1:os%sphere%np))
    do idir = 1, space%dim
      do is = 1, os%sphere%np
        x_submesh(is) = os%sphere%rel_x(idir, is) + os%sphere%center(idir)
      end do

      do im = 1, os%norbs
        !In case of phase, we have to apply the conjugate of the phase here
        if (psib%has_phase) then
#ifdef R_TCOMPLEX
          if (.not. this%basis%use_submesh) then
            !If we orthogonalize, the orbital is not anymore zorb*phase
            if (.not. this%basis%orthogonalization) then
              do idim = 1, d%dim
                idim_orb = min(idim,os%ndim)
                epsi(:,idim) = R_TOTYPE(M_ZERO)
                do is = 1, os%sphere%np
                  epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) &
                    + x_submesh(is)*os%zorb(is,idim_orb,im)*os%phase(is,psib%ik)
                end do
              end do
            else
              do idim = 1, d%dim
                idim_orb = min(idim,os%ndim)
                !$omp parallel do
                do is = 1, mesh%np
                  epsi(is,idim) = os%sphere%mesh%x(is, idir)*os%eorb_mesh(is, im, idim_orb, psib%ik)
                end do
              end do
            end if
          else
            do idim = 1, d%dim
              epsi(:,idim) = R_TOTYPE(M_ZERO)
              idim_orb = min(idim,os%ndim)
              do is = 1, os%sphere%np
                epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) + x_submesh(is)*os%eorb_submesh(is,idim_orb,im,psib%ik)
              end do
            end do
          end if
#endif
        else
          if (.not. use_submesh) then
            do idim = 1, d%dim
              idim_orb = min(idim,os%ndim)
              !$omp parallel do
              do is = 1, mesh%np
                epsi(is,idim) = os%sphere%mesh%x(is, idir)*os%X(orb)(is, idim_orb, im)
              end do
            end do
          else
            do idim = 1, d%dim
              epsi(:,idim) = R_TOTYPE(M_ZERO)
              idim_orb = min(idim,os%ndim)
              do is = 1, os%sphere%np
                epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) + x_submesh(is)*os%X(orb)(is, idim_orb, im)
              end do
            end do
          end if
        end if
        call X(batch_ax_function_py)(mesh%np, weight(im, :, ios), epsi, gpsib(idir))
      end do !im
    end do !idir
    SAFE_DEALLOCATE_A(x_submesh)
  end do !ios

  SAFE_DEALLOCATE_A(epsi)


  do idir = 1, space%dim
    weight = M_ZERO
    phi_dot_psi = M_ZERO

    ! We have to compute
    ! hpsi> -= sum_m |phi m> sum_mp Vmmp <phi mp| r | psi >
    !
    ! We first compute <phi m| r | psi> for all orbitals of the atom
    !
    ! The non-periodic case is simple as we can efficiently compute
    ! x|\psi> and get the coefficient
    ! For the periodic case, this is more involved
    !
    if(.not. space%is_periodic()) then

      ! Only works for the non-periodic case
      call psib%copy_to(rpsib, copy_data = .false.)
      call batch_mul(mesh%np, mesh%x(:,idir), psib, rpsib)
      do ios = 1, this%norbsets
        os => this%orbsets(ios)
        call X(orbitalset_get_coeff_batch)(os, d%dim, rpsib, phi_dot_psi(:,:,:,ios))
      end do
      call rpsib%end(copy = .false.)

    else
      do ios = 1, this%norbsets
        os => this%orbsets(ios)
        call X(orbitalset_get_position_matrix_elem)(os, d%dim, psib, idir, phi_dot_psi(:,:,:,ios))
      end do
    end if

    do ios = 1, this%norbsets
      ! We now compute weight = sum_mp Vmmp <phi mp | psi >
      os => this%orbsets(ios)
      !
      call X(dft_u_get_weight_projectors)(this, os, ios, d%ispin, psib, phi_dot_psi, &
        weight(1:os%norbs, 1:psib%nst_linear, ios))
    end do !ios

    do ios = 1, this%norbsets
      os => this%orbsets(ios)
      call X(orbitalset_add_to_batch)(os, d%dim, gpsib(idir), -weight(:,:,ios))
    end do
  end do !idir

  SAFE_DEALLOCATE_A(phi_dot_psi)

  POP_SUB(lda_u_commute_r)
  call profiling_out(TOSTRING(X(DFTU_COMMUTE_R)))
end subroutine X(lda_u_commute_r)

! ---------------------------------------------------------
subroutine X(lda_u_force)(this, namespace, space, mesh, st, iq, psib, grad_psib, force, phase)
  type(lda_u_t),             intent(in)    :: this
  type(namespace_t),         intent(in)    :: namespace
  class(space_t),            intent(in)    :: space
  class(mesh_t),             intent(in)    :: mesh
  type(states_elec_t),       intent(in)    :: st
  integer,                   intent(in)    :: iq
  type(wfs_elec_t),          intent(in)    :: psib
  type(wfs_elec_t),          intent(in)    :: grad_psib(:)
  real(real64),              intent(inout) :: force(:, :)
  logical,                   intent(in)    :: phase

  integer :: ios, iatom, ibatch, ist, im, imp, ispin, idir
  type(orbitalset_t), pointer  :: os
  R_TYPE :: ff(1:space%dim)
  R_TYPE, allocatable :: psi(:,:), gpsi(:,:)
  R_TYPE, allocatable :: phi_dot_psi(:,:,:), phi_dot_gpsi(:,:,:,:), gradn(:,:,:,:)
  real(real64) :: weight

  if (this%level == DFT_U_NONE) return
  !In this case, there is no contribution to the force
  if (this%basisfromstates) return

  if (this%double_couting /= DFT_U_FLL) then
    call messages_not_implemented("AMF double couting with forces", namespace=namespace)
  end if

  PUSH_SUB(X(lda_u_force))

  call profiling_in(TOSTRING(X(FORCES_DFTU)))

  !TODO: Implement
  if (this%intersite) then
    message(1) = "Intersite V forces are not implemented."
    call messages_warning(1, namespace=namespace)
  end if

  SAFE_ALLOCATE(psi(1:mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(gpsi(1:mesh%np, 1:st%d%dim))
  SAFE_ALLOCATE(phi_dot_psi(1:st%d%dim, 1:this%maxnorbs, 1:psib%nst))
  SAFE_ALLOCATE(phi_dot_gpsi(1:st%d%dim, 1:this%maxnorbs, 1:psib%nst, 1:space%dim))
  SAFE_ALLOCATE(gradn(1:this%maxnorbs,1:this%maxnorbs,1:this%nspins,1:space%dim))

  ispin = st%d%get_spin_index(iq)

  do ios = 1, this%norbsets
    os => this%orbsets(ios)
    iatom = os%iatom

    gradn(1:os%norbs,1:os%norbs,1:this%nspins,1:space%dim) = R_TOTYPE(M_ZERO)

    !We first compute the matrix elemets <orb_m |\psi>
    !taking into account phase correction if needed
    !
    !No phase here, this is already added
    call X(orbitalset_get_coeff_batch)(os, st%d%dim, psib, phi_dot_psi(:, :, 1:psib%nst))

    do idir = 1, space%dim
      call X(orbitalset_get_coeff_batch)(os, st%d%dim, grad_psib(idir), phi_dot_gpsi(:, :, 1:psib%nst, idir))
    end do

    do ibatch = 1, psib%nst
      ist = psib%ist(ibatch)
      weight = st%kweights(iq)*st%occ(ist, iq)
      if (weight < 1.0e-10_real64) cycle

      do idir = 1, space%dim
        if (st%d%ispin /= SPINORS) then
          do im = 1, os%norbs
            gradn(1:os%norbs, im, ispin, idir) = gradn(1:os%norbs, im, ispin, idir) &
              + weight*(phi_dot_gpsi(1, 1:os%norbs, ibatch, idir)*R_CONJ(phi_dot_psi(1, im, ibatch)) &
              +R_CONJ(phi_dot_gpsi(1, im, ibatch, idir))*phi_dot_psi(1, 1:os%norbs, ibatch))
          end do
        else
          do im = 1, os%norbs
            do ispin = 1, this%spin_channels
              gradn(1:os%norbs, im, ispin, idir) = gradn(1:os%norbs, im, ispin, idir) &
                + weight*(phi_dot_gpsi(ispin, 1:os%norbs, ibatch, idir)*R_CONJ(phi_dot_psi(ispin, im, ibatch)) &
                +R_CONJ(phi_dot_gpsi(ispin, im, ibatch, idir))*phi_dot_psi(ispin, 1:os%norbs, ibatch))
            end do
            gradn(1:os%norbs, im, 3, idir) = gradn(1:os%norbs, im, 3, idir) &
              + weight*(phi_dot_gpsi(1, 1:os%norbs, ibatch, idir)*R_CONJ(phi_dot_psi(2, im, ibatch)) &
              +R_CONJ(phi_dot_gpsi(2, im, ibatch, idir))*phi_dot_psi(1, 1:os%norbs, ibatch))
            gradn(1:os%norbs, im, 4, idir) = gradn(1:os%norbs, im, 4, idir) &
              + weight*(phi_dot_gpsi(2, 1:os%norbs, ibatch, idir)*R_CONJ(phi_dot_psi(1, im, ibatch)) &
              +R_CONJ(phi_dot_gpsi(1, im, ibatch, idir))*phi_dot_psi(2, 1:os%norbs, ibatch))
          end do

        end if
      end do !idir

    end do !ibatch

    if (st%d%ispin /= SPINORS) then
      ff(1:space%dim) = M_ZERO
      do im = 1, os%norbs
        do imp = 1, os%norbs
          ff(1:space%dim) = ff(1:space%dim) - this%X(n)(imp,im,ispin,ios)/st%smear%el_per_state*gradn(im,imp,ispin,1:space%dim)
        end do !imp
        ff(1:space%dim) = ff(1:space%dim) + M_HALF*gradn(im, im, ispin,1:space%dim)
      end do !im
    else
      ff(1:space%dim) = M_ZERO
      do ispin = 1, st%d%nspin
        do im = 1, os%norbs
          do imp = 1, os%norbs
            !We use R_CONJ to get n(imp,im, sigmap, sigma) from n(im,imp, sigma,sigmap)
            ff(1:space%dim) = ff(1:space%dim) - R_CONJ(this%X(n)(im,imp,ispin,ios))/st%smear%el_per_state &
              *gradn(im,imp,ispin,1:space%dim)
          end do !imp
          if (ispin <= this%spin_channels) then
            ff(1:space%dim) = ff(1:space%dim) + M_HALF*gradn(im, im, ispin,1:space%dim)
          end if
        end do !im
      end do !ispin
    end if


    force(1:space%dim, iatom) = force(1:space%dim, iatom) - os%Ueff*real(ff(1:space%dim), real64)
  end do !ios

  SAFE_DEALLOCATE_A(psi)
  SAFE_DEALLOCATE_A(gpsi)
  SAFE_DEALLOCATE_A(phi_dot_psi)
  SAFE_DEALLOCATE_A(phi_dot_gpsi)
  SAFE_DEALLOCATE_A(gradn)

  call profiling_out(TOSTRING(X(FORCES_DFTU)))

  POP_SUB(X(lda_u_force))
end subroutine X(lda_u_force)

! ---------------------------------------------------------
!>@brief This routine computes \f$(r-R_J)V_{\rm DFT+U}|\psi_{n,\mathbf{k}}\rangle\f$
!!
!! @note
!! The notation is weird and needs to be understood as the fact that the factor
!! \f$(r_R_J)\f$ is in fact included in the Bloch sum of the left part
!! of the projector over the orbitals.
subroutine X(lda_u_rvu)(this, mesh, space, d, namespace, psib, gpsib)
  type(lda_u_t),           intent(in) :: this
  class(mesh_t),           intent(in) :: mesh
  class(space_t),          intent(in) :: space
  type(states_elec_dim_t), intent(in) :: d
  type(namespace_t),       intent(in) :: namespace
  type(wfs_elec_t),        intent(in) :: psib
  class(wfs_elec_t),    intent(inout) :: gpsib(:)

  integer :: ios, idim, idir, im, is
  integer :: idim_orb
  R_TYPE, allocatable :: orb_dot_psib(:,:,:,:), projection_weight(:,:,:), epsi(:,:)
  type(orbitalset_t), pointer  :: os
  logical :: use_submesh
  real(real64), allocatable :: x_submesh(:)

  if (this%level == DFT_U_NONE) return

  PUSH_SUB_WITH_PROFILE(lda_u_rvu)

  if (this%double_couting /= DFT_U_FLL) then
    call messages_not_implemented("Stress tensor with AMF double counting", namespace=namespace)
  end if

  if (this%intersite) then
    call messages_not_implemented("Stress tensor with DFT+U+V", namespace=namespace)
  end if

  use_submesh = this%basis%use_submesh
  ! Because of possible phase corrections at the border, the array X(orb) is always stored
  ! on the submesh for complex wavefunctions
  ! This does only apply to X(orb). eorb_mesh/eorb_submesh are
  ! still stored according to the user choice given by basis%submesh.
  ! Hence, only if we do not have phases but have a complex wavefunctions we will access X(orb)
  ! always on the submesh
#ifdef R_TCOMPLEX
  if (.not. psib%has_phase) use_submesh = .true.
#endif

  SAFE_ALLOCATE(orb_dot_psib(1:d%dim,1:this%maxnorbs, 1:psib%nst, 1:this%norbsets))
  SAFE_ALLOCATE(projection_weight(1:this%maxnorbs, 1:psib%nst_linear, 1:this%norbsets))

  do ios = 1, this%norbsets
    ! We have to compute
    ! gpsi> += r sum_m |phi m> sum_mp Vmmp <phi mp | psi >
    !
    ! We first compute <phi m | psi> for all orbitals of the atom
    !
    os => this%orbsets(ios)
    !
    call X(orbitalset_get_coeff_batch)(os, d%dim, psib, orb_dot_psib(:,:,:,ios))
  end do

  projection_weight(:,:,:) = M_ZERO

  do ios = 1, this%norbsets
    ! We now compute projection_weight = sum_mp Vmmp <phi mp | psi >
    os => this%orbsets(ios)

    call X(dft_u_get_weight_projectors)(this, os, ios, d%ispin, psib, orb_dot_psib, &
      projection_weight(1:os%norbs, 1:psib%nst_linear, ios))
  end do !ios

  SAFE_ALLOCATE(epsi(1:mesh%np, 1:d%dim))

  do ios = 1, this%norbsets
    os => this%orbsets(ios)

    SAFE_ALLOCATE(x_submesh(1:os%sphere%np))
    do idir = 1, space%dim
      do is = 1, os%sphere%np
        x_submesh(is) = os%sphere%rel_x(idir, is)
      end do

      do im = 1, os%norbs
        do idim = 1, d%dim
          idim_orb = min(idim,os%ndim)
          !In case of phase, we have to apply the conjugate of the phase here
          if (psib%has_phase) then
#ifdef R_TCOMPLEX
            if (.not. this%basis%use_submesh) then
              !If we orthogonalize, the orbital is not anymore zorb*phase
              if (.not. this%basis%orthogonalization) then
                epsi(:,idim) = R_TOTYPE(M_ZERO)
                do is = 1, os%sphere%np
                  epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) &
                    + x_submesh(is)*os%zorb(is,idim_orb,im)*os%phase(is,psib%ik)
                end do
              else
                !$omp parallel do
                do is = 1, mesh%np
                  epsi(is,idim) = (os%sphere%mesh%x(is, idir)-os%sphere%center(idir))*os%eorb_mesh(is, im, idim_orb, psib%ik)
                end do
              end if
            else
              epsi(:,idim) = R_TOTYPE(M_ZERO)
              do is = 1, os%sphere%np
                epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) + x_submesh(is)*os%eorb_submesh(is,idim_orb,im,psib%ik)
              end do
            end if
#endif
          else
            if (.not. use_submesh) then
              !$omp parallel do
              do is = 1, mesh%np
                epsi(is,idim) = (os%sphere%mesh%x(is, idir)-os%sphere%center(idir))*os%X(orb)(is, idim_orb, im)
              end do
            else
              epsi(:,idim) = R_TOTYPE(M_ZERO)
              do is = 1, os%sphere%np
                epsi(os%sphere%map(is),idim) = epsi(os%sphere%map(is),idim) + x_submesh(is)*os%X(orb)(is, idim_orb, im)
              end do
            end if
          end if
        end do !idim
        call X(batch_ax_function_py)(mesh%np, projection_weight(im, :, ios), epsi, gpsib(idir))
      end do !im
    end do !idir
    SAFE_DEALLOCATE_A(x_submesh)
  end do !ios

  SAFE_DEALLOCATE_A(epsi)

  SAFE_DEALLOCATE_A(orb_dot_psib)
  SAFE_DEALLOCATE_A(projection_weight)

  POP_SUB_WITH_PROFILE(lda_u_rvu)
end subroutine X(lda_u_rvu)


 ! ---------------------------------------------------------
subroutine X(lda_u_set_occupations)(this, occ)
  type(lda_u_t),  intent(inout) :: this
  R_TYPE,         intent(in)    :: occ(:)

  integer :: ios, ispin, im, imp, ind, norbs, inn, ios2

  PUSH_SUB(X(lda_u_set_occupations))

  ind = 0
  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    do ispin = 1, this%nspins
      do im = 1, norbs
        do imp = 1,norbs
          ind = ind + 1
          this%X(n)(im,imp,ispin,ios) = occ(ind)
        end do
      end do
    end do
  end do

  if (this%level == DFT_U_ACBN0) then
    do ios = 1, this%norbsets
      norbs = this%orbsets(ios)%norbs
      do ispin = 1, this%nspins
        do im = 1, norbs
          do imp = 1,norbs
            ind = ind + 1
            this%X(n_alt)(im,imp,ispin,ios) = occ(ind)
          end do
        end do
      end do
    end do

    if (this%intersite) then
      do ios = 1, this%norbsets
        do inn = 1, this%orbsets(ios)%nneighbors
          ios2 = this%orbsets(ios)%map_os(inn)
          do ispin = 1, this%nspins
            do im = 1, this%orbsets(ios)%norbs
              do imp = 1,this%orbsets(ios2)%norbs
                ind = ind + 1
                this%X(n_ij)(im,imp,ispin,ios,inn) = occ(ind)
                ind = ind + 1
                this%X(n_alt_ij)(im,imp,ispin,ios,inn) = occ(ind)
              end do
            end do
          end do
        end do
      end do
    end if
  end if

  POP_SUB(X(lda_u_set_occupations))
end subroutine X(lda_u_set_occupations)

 ! ---------------------------------------------------------
subroutine X(lda_u_get_occupations)(this, occ)
  type(lda_u_t),  intent(in) :: this
  R_TYPE,      intent(inout) :: occ(:)

  integer :: ios, ispin, im, imp, ind, norbs, inn, ios2

  PUSH_SUB(X(lda_u_get_occupations))

  ind = 0
  do ios = 1, this%norbsets
    norbs = this%orbsets(ios)%norbs
    do ispin = 1, this%nspins
      do im = 1, norbs
        do imp = 1,norbs
          ind = ind + 1
          occ(ind) = this%X(n)(im,imp,ispin,ios)
        end do
      end do
    end do
  end do

  if (this%level == DFT_U_ACBN0) then
    do ios = 1, this%norbsets
      norbs = this%orbsets(ios)%norbs
      do ispin = 1, this%nspins
        do im = 1, norbs
          do imp = 1,norbs
            ind = ind + 1
            occ(ind) = this%X(n_alt)(im,imp,ispin,ios)
          end do
        end do
      end do
    end do

    if (this%intersite) then
      do ios = 1, this%norbsets
        do inn = 1, this%orbsets(ios)%nneighbors
          ios2 = this%orbsets(ios)%map_os(inn)
          do ispin = 1, this%nspins
            do im = 1, this%orbsets(ios)%norbs
              do imp = 1,this%orbsets(ios2)%norbs
                ind = ind + 1
                occ(ind) = this%X(n_ij)(im,imp,ispin,ios,inn)
                ind = ind + 1
                occ(ind) = this%X(n_alt_ij)(im,imp,ispin,ios,inn)
              end do
            end do
          end do
        end do
      end do
    end if
  end if

  POP_SUB(X(lda_u_get_occupations))
end subroutine X(lda_u_get_occupations)

 ! ---------------------------------------------------------
subroutine X(lda_u_allocate)(this, st)
  type(lda_u_t),  intent(inout)  :: this
  type(states_elec_t), intent(in):: st

  integer :: maxorbs, nspin

  PUSH_SUB(X(lda_u_allocate))

  maxorbs = this%maxnorbs
  nspin = this%nspins

  SAFE_ALLOCATE(this%X(n)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets))
  this%X(n)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets) = R_TOTYPE(M_ZERO)
  SAFE_ALLOCATE(this%X(V)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets))
  this%X(V)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets) = R_TOTYPE(M_ZERO)

  !In case we use the ab-initio scheme, we need to allocate extra resources
  if (this%level == DFT_U_ACBN0) then
    SAFE_ALLOCATE(this%X(n_alt)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets))
    this%X(n_alt)(1:maxorbs,1:maxorbs,1:nspin,1:this%norbsets) = R_TOTYPE(M_ZERO)
    SAFE_ALLOCATE(this%renorm_occ(this%nspecies,0:7,0:(MAX_L-1),st%st_start:st%st_end,st%d%kpt%start:st%d%kpt%end))
    this%renorm_occ(this%nspecies,0:7,0:(MAX_L-1),st%st_start:st%st_end,st%d%kpt%start:st%d%kpt%end) = M_ZERO
  end if

  POP_SUB(X(lda_u_allocate))
end subroutine X(lda_u_allocate)


! ---------------------------------------------------------
!>@brief Given the dot products  \f$<phi mp | psi >\f$, this computes the weight of the DFT+U projectors
!!
!! For DFT+U+V (intersite interaction), the weight also include the intersite part, such that the resulting
!! weight is applied only once to the localized orbitals.
!!
!! For the spinor case, the weight is defined per spin component, so the last dimension of weight is of size nst_linear
subroutine X(dft_u_get_weight_projectors)(this, os, ios, spin_component, psib, phi_dot_psi, weight)
  type(lda_u_t),        intent(in)    :: this
  type(orbitalset_t),   intent(in)    :: os
  integer,              intent(in)    :: ios
  integer,              intent(in)    :: spin_component
  type(wfs_elec_t),     intent(in)    :: psib
  R_TYPE,               intent(in)    :: phi_dot_psi(:,:,:,:) !< Dot products \f$<phi mp | psi >\f$ (nspin, norb, nst, norbset)
  R_TYPE,               intent(inout) :: weight(:,:) !< Resulting weight (norb, nst_linear)

  integer :: ist, bind1, bind2, im, imp, ios2, inn, el_per_state, ispin
  R_TYPE :: v_ij

  PUSH_SUB(X(dft_u_get_weight_projectors))

  select case(spin_component)
  case(UNPOLARIZED)
    el_per_state = 2
    ispin = 1
  case(SPIN_POLARIZED)
    el_per_state = 1
    ispin = 1 + mod(psib%ik - 1, 2) ! See states_elec_dim_get_spin_index
  case(SPINORS)
    el_per_state = 1
    ispin = 1
  end select

  do ist = 1, psib%nst
    bind1 = psib%ist_idim_to_linear((/ist, 1/))
    bind2 = psib%ist_idim_to_linear((/ist, 2/))
    ! sum_mp Vmmp <phi mp | psi >
    do imp = 1, os%norbs
      do im = 1, os%norbs
        if (spin_component /= SPINORS) then
          weight(im, ist) = weight(im, ist) + this%X(V)(im, imp, ispin, ios)*phi_dot_psi(1, imp, ist, ios)
        else
          weight(im, bind1) = weight(im, bind1) + this%X(V)(im, imp, 1, ios) * phi_dot_psi(1, imp, ist, ios)
          weight(im, bind1) = weight(im, bind1) + this%X(V)(im, imp, 3, ios) * phi_dot_psi(2, imp, ist, ios)
          weight(im, bind2) = weight(im, bind2) + this%X(V)(im, imp, 4, ios) * phi_dot_psi(1, imp, ist, ios)
          weight(im, bind2) = weight(im, bind2) + this%X(V)(im, imp, 2, ios) * phi_dot_psi(2, imp, ist, ios)
        end if
      end do !im
    end do !imp
  end do !ist

  !We add the intersite interaction
  if (this%intersite) then
    !Loop over nearest neighbors
    do inn = 1, os%nneighbors
      ios2 = os%map_os(inn)

      v_ij = os%V_ij(inn, 0)/el_per_state
      if (psib%has_phase) then
#ifdef R_TCOMPLEX
        v_ij = v_ij * os%phase_shift(inn, psib%ik)
#else
        ASSERT(.false.) !Phase can only be applied to complex wavefunctions
#endif
      end if

      do ist = 1, psib%nst
        bind1 = psib%ist_idim_to_linear((/ist, 1/))
        bind2 = psib%ist_idim_to_linear((/ist, 2/))
        do imp = 1, this%orbsets(ios2)%norbs
          do im = 1,os%norbs
            if (spin_component /= SPINORS) then
              weight(im, ist) = weight(im, ist) - phi_dot_psi(1, imp, ist, ios2) &
                * this%X(n_ij)(im, imp, ispin, ios, inn) * v_ij
            else! Spinors
              weight(im, bind1) = weight(im, bind1) - phi_dot_psi(1, imp, ist, ios2) &
                * this%X(n_ij)(im, imp, 1, ios, inn) * v_ij
              weight(im, bind1) = weight(im, bind1) - phi_dot_psi(2, imp, ist, ios2) &
                * this%X(n_ij)(im, imp, 3, ios, inn) * v_ij
              weight(im, bind2) = weight(im, bind2) - phi_dot_psi(1, imp, ist, ios2) &
                * this%X(n_ij)(im, imp, 4, ios, inn) * v_ij
              weight(im, bind2) = weight(im, bind2) - phi_dot_psi(2, imp, ist, ios2) &
                * this%X(n_ij)(im, imp, 2, ios, inn) * v_ij
            end if
          end do !im
        end do !imp
      end do !ist
    end do !inn
  end if

  POP_SUB(X(dft_u_get_weight_projectors))
end subroutine X(dft_u_get_weight_projectors)

