!! Copyright (C) 2010 X. Andrade
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module projector_matrix_oct_m
  use debug_oct_m
  use global_oct_m
  use messages_oct_m
  use profiling_oct_m
  use submesh_oct_m

  implicit none

  private

  public ::                     &
    projector_matrix_t,         &
    projector_matrix_allocate,  &
    projector_matrix_deallocate

  !> A set of projectors defined on a submesh
  type projector_matrix_t
    ! Components are public by default
    real(real64),   allocatable :: dprojectors(:, :)
    complex(real64),   allocatable :: zprojectors(:, :)
    real(real64),   allocatable :: scal(:)
    integer              :: npoints !< submesh%np
    integer              :: nprojs
    integer              :: nregions
    integer, allocatable :: regions(:)
    real(real64),   allocatable :: dmix(:, :)
    complex(real64),   allocatable :: zmix(:, :, :)
    logical              :: is_cmplx = .false.

    integer, pointer, contiguous :: map(:) !< Points towards submesh%map
    real(real64),   pointer, contiguous :: position(:, :) !< Points towards submesh%rel_x
  end type projector_matrix_t

contains

  ! -------------------------------------------------
  subroutine projector_matrix_allocate(this, nprojs, sphere, has_mix_matrix, is_cmplx)
    type(projector_matrix_t), intent(out) :: this
    integer,                  intent(in)  :: nprojs
    type(submesh_t),          intent(in)  :: sphere
    logical,                  intent(in)  :: has_mix_matrix
    logical, optional,        intent(in)  :: is_cmplx

    PUSH_SUB(projector_matrix_allocate)

    this%npoints = sphere%np
    this%nprojs = nprojs
    this%nregions = sphere%num_regions

    this%is_cmplx = optional_default(is_cmplx, .false.)

    SAFE_ALLOCATE(this%regions(1:this%nregions+1))
    if (this%is_cmplx) then
      SAFE_ALLOCATE(this%zprojectors(1:this%npoints, 1:nprojs))
    else
      SAFE_ALLOCATE(this%dprojectors(1:this%npoints, 1:nprojs))
    end if
    SAFE_ALLOCATE(this%scal(1:nprojs))

    if (has_mix_matrix) then
      if (this%is_cmplx) then
        SAFE_ALLOCATE(this%zmix(1:nprojs, 1:nprojs, 1:4))
      else
        SAFE_ALLOCATE(this%dmix(1:nprojs, 1:nprojs))
      end if
    end if

    POP_SUB(projector_matrix_allocate)
  end subroutine projector_matrix_allocate

  ! -------------------------------------------------

  subroutine projector_matrix_deallocate(this)
    type(projector_matrix_t), intent(inout) :: this

    PUSH_SUB(projector_matrix_deallocate)

    SAFE_DEALLOCATE_A(this%regions)
    SAFE_DEALLOCATE_A(this%dprojectors)
    SAFE_DEALLOCATE_A(this%zprojectors)
    SAFE_DEALLOCATE_A(this%scal)
    SAFE_DEALLOCATE_A(this%dmix)
    SAFE_DEALLOCATE_A(this%zmix)

    nullify(this%position)
    nullify(this%map)

    POP_SUB(projector_matrix_deallocate)
  end subroutine projector_matrix_deallocate

  ! -------------------------------------------------

end module projector_matrix_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
