!! Copyright (C) 2023 S. Ohlmann
!!
!! This Source Code Form is subject to the terms of the Mozilla Public
!! License, v. 2.0. If a copy of the MPL was not distributed with this
!! file, You can obtain one at https://mozilla.org/MPL/2.0/.
!!
#include "global.h"

!> @brief This module implements the field transfer
!!
!! Idea: field_transfer_t is an abstract interaction that offers regridding
!! and interpolation and can be subclassed for concrete implementations
!! (currently current_to_mxll_field_oct_m::current_to_mxll_field_t
!! and mxll_e_field_to_matter_oct_m::mxll_e_field_to_matter_t).
!!
!! ## How to get quantities from a field transfer interaction using the interpolation
!!
!! In order to get correctly interpolated values also when the systems have different timesteps,
!! at the moment, one always needs to call the interpolate function to get the field at the requested time.
!! As an example from the Maxwell system, this is how to get the current at a certain time:
!! ```Fortran
!! ! interpolate current from interaction
!! call iter%start(this%interactions)
!! do while (iter%has_next())
!!   select type (interaction => iter%get_next())
!!   class is (current_to_mxll_field_t)
!!     call interaction%interpolate(time, current_density_ext)
!!     call lalg_axpy(this%gr%np, 3, M_ONE, current_density_ext * this%hm%current_factor, current)
!!   end select
!! end do
!! ```
!! One can loop over all interactions, call the interpolate function and add
!! the result to get the total current in this case.
!!
!! ## Time interpolation
!!
!! The time interpolation class (time_interpolation_t) offers a interpolation in time
!! for each point of a field for an arbitrary depth. It uses the math_oct_m::interpolate function
!! from the math_oct_m module that is based on Lagrange polynomial interpolation.
!! The time interpolation class implements also functionality for reading and writing
!! restart data which is needed to be able to restart consistently with the same
!! history of past values.
!!
!! Derived interactions
!!
!! - mxll_e_field_to_matter_oct_m::mxll_e_field_to_matter_t
!! - current_to_mxll_field_oct_m::current_to_mxll_field_t

module field_transfer_oct_m
  use debug_oct_m
  use global_oct_m
  use grid_oct_m
  use interaction_oct_m
  use interaction_partner_oct_m
  use mesh_oct_m
  use messages_oct_m
  use namespace_oct_m
  use profiling_oct_m
  use quantity_oct_m
  use regridding_oct_m
  use restart_oct_m
  use states_mxll_oct_m
  use time_interpolation_oct_m

  implicit none

  private
  public :: &
    field_transfer_t,   &
    field_transfer_init

  !> @brief class defining the field_transfer interaction
  !!
  !! It contains the field from the partner (partner_field) and the one transferred to the system (system_field).
  !! Moreover, it contains data structures for regridding and time interpolation.
  type, extends(interaction_t), abstract :: field_transfer_t
    private
    real(real64), allocatable, public     :: partner_field(:,:)   !< field from partner
    real(real64), allocatable, public     :: system_field(:,:)    !< field transferred to system grid
    type(grid_t), pointer, public  :: system_gr  => NULL() !< pointer to grid of the system
    type(regridding_t), pointer, public :: regridding => NULL()
    type(time_interpolation_t), pointer, public :: interpolation => NULL()
    integer, public :: ndim
    logical, public :: interpolation_initialized = .false.

  contains
    procedure :: init => field_transfer_init
    !< @copydoc field_transfer_oct_m::field_transfer_init
    procedure :: init_from_partner => field_transfer_init_from_partner
    !< @copydoc field_transfer_oct_m::field_transfer_init_from_partner
    procedure :: init_interpolation => field_transfer_init_interpolation
    !< @copydoc field_transfer_oct_m::field_transfer_init_interpolation
    procedure :: calculate => field_transfer_calculate
    !< @copydoc field_transfer_oct_m::field_transfer_calculate
    procedure :: do_mapping => field_transfer_do_mapping
    !< @copydoc field_transfer_oct_m::field_transfer_do_mapping
    procedure :: dfield_transfer_interpolate, zfield_transfer_interpolate
    generic :: interpolate => dfield_transfer_interpolate, zfield_transfer_interpolate
    !< @copydoc field_transfer_oct_m::dfield_transfer_interpolate
    procedure :: calculate_energy => field_transfer_calculate_energy
    !< @copydoc field_transfer_oct_m::field_transfer_calculate_energy
    procedure :: read_restart => field_transfer_read_restart
    !< @copydoc field_transfer_oct_m::field_transfer_read_restart
    procedure :: write_restart => field_transfer_write_restart
    !< @copydoc field_transfer_oct_m::field_transfer_write_restart
    procedure :: end => field_transfer_end
    !< @copydoc field_transfer_oct_m::field_transfer_end
  end type field_transfer_t

contains

  !> the system field is allocated and initialized to 0
  subroutine field_transfer_init(this, gr, ndim)
    class(field_transfer_t), intent(inout) :: this
    type(grid_t), target,    intent(in)    :: gr
    integer,                 intent(in)    :: ndim

    PUSH_SUB(field_transfer_init)

    this%system_gr => gr
    this%ndim = ndim
    SAFE_ALLOCATE(this%system_field(1:gr%np, 1:ndim))
    this%system_field(:,:) = M_zero

    POP_SUB(field_transfer_init)
  end subroutine field_transfer_init

  !>  the partner field is allocated and initialized to 0; moreover the regridding structure is initialized
  subroutine field_transfer_init_from_partner(this, partner_gr, partner_space, partner_namespace)
    class(field_transfer_t), intent(inout) :: this
    type(grid_t),            intent(in)    :: partner_gr
    class(space_t),          intent(in)    :: partner_space
    type(namespace_t),       intent(in)    :: partner_namespace

    PUSH_SUB(field_transfer_init_from_partner)

    SAFE_ALLOCATE(this%partner_field(1:partner_gr%np, 1:this%ndim))
    this%partner_field(:,:) = M_zero
    this%regridding => regridding_t(this%system_gr, partner_gr, partner_space, partner_namespace)

    POP_SUB(field_transfer_init_from_partner)
  end subroutine field_transfer_init_from_partner

  !> the time interpolation is initialized; it needs to know the depth which is usually
  !! given by the order of the propagator; thus it can only be called after the algorithm
  !! is known (this can be done in the system in new_algorithm)
  subroutine field_transfer_init_interpolation(this, depth, label, cmplx)
    class(field_transfer_t), intent(inout) :: this
    integer,                 intent(in)    :: depth
    character(len=*),        intent(in)    :: label
    logical, optional,       intent(in)    :: cmplx

    PUSH_SUB(field_transfer_init_interpolation)

    this%interpolation => time_interpolation_t(this%system_gr%np, this%ndim, depth, &
      optional_default(cmplx, .false.), trim(label) // '-' // trim(this%partner%namespace%get()))
    this%interpolation_initialized = .true.

    POP_SUB(field_transfer_init_interpolation)
  end subroutine field_transfer_init_interpolation

  subroutine field_transfer_end(this)
    class(field_transfer_t), intent(inout) :: this

    PUSH_SUB(field_transfer_end)

    call interaction_end(this)
    SAFE_DEALLOCATE_A(this%partner_field)
    SAFE_DEALLOCATE_A(this%system_field)
    SAFE_DEALLOCATE_P(this%regridding)
    SAFE_DEALLOCATE_P(this%interpolation)

    POP_SUB(field_transfer_end)
  end subroutine field_transfer_end

  subroutine field_transfer_calculate(this)
    class(field_transfer_t), intent(inout) :: this

    PUSH_SUB(field_transfer_calculate)

    ! nothing to do here
    ! the mapping is done when the quantity is copied to the interaction
    ! for this interaction, we do not need input from the system, so the
    ! values do only change when the partner quantity changes

    POP_SUB(field_transfer_calculate)
  end subroutine field_transfer_calculate

  !> @brief perform the regridding and add the system field to the
  !! time interpolator using the time of the quantity at this point;
  !!
  !! it is called by the partner system when copying its data to
  !! the interaction because the system grid does not change
  !! Note: this needs an extra function call for each of the partner
  !! systems in their copy_quantities_to_interaction function
  !!
  subroutine field_transfer_do_mapping(this)
    class(field_transfer_t), intent(inout) :: this

    PUSH_SUB_WITH_PROFILE(field_transfer_do_mapping)

    ASSERT(this%interpolation_initialized)

    if (allocated(this%partner_field)) then
      call this%regridding%do_transfer(this%system_field, this%partner_field)
    end if

    associate(coupling => this%partner%quantities%get(this%couplings_from_partner(1)))
      call this%interpolation%add_time(coupling%iteration%value(), this%system_field)
    end associate

    POP_SUB_WITH_PROFILE(field_transfer_do_mapping)
  end subroutine field_transfer_do_mapping

  !> return the interpolated field for a given time
  subroutine dfield_transfer_interpolate(this, time, field)
    class(field_transfer_t),  intent(in)  :: this
    real(real64),             intent(in)  :: time
    real(real64), contiguous, intent(out) :: field(:, :)


    PUSH_SUB_WITH_PROFILE(dfield_transfer_interpolate)

    call this%interpolation%interpolate(time, field)

    POP_SUB_WITH_PROFILE(dfield_transfer_interpolate)
  end subroutine dfield_transfer_interpolate

  !> return the interpolated field for a given time
  subroutine zfield_transfer_interpolate(this, time, field)
    class(field_transfer_t),     intent(in)  :: this
    real(real64),                intent(in)  :: time
    complex(real64), contiguous, intent(out) :: field(:, :)


    PUSH_SUB_WITH_PROFILE(zfield_transfer_interpolate)

    call this%interpolation%interpolate(time, field)

    POP_SUB_WITH_PROFILE(zfield_transfer_interpolate)
  end subroutine zfield_transfer_interpolate

  subroutine field_transfer_calculate_energy(this)
    class(field_transfer_t),    intent(inout) :: this

    PUSH_SUB(field_transfer_calculate_energy)

    ! interaction energy is zero, since it is only re-gridding the quantities of one system
    ! on the mesh of the other
    this%energy = M_ZERO

    POP_SUB(field_transfer_calculate_energy)
  end subroutine field_transfer_calculate_energy

  subroutine field_transfer_read_restart(this, mesh, space, restart, err)
    class(field_transfer_t), intent(inout) :: this
    class(mesh_t),           intent(in)    :: mesh
    class(space_t),          intent(in)    :: space
    type(restart_t),         intent(in)    :: restart
    integer,                 intent(out)   :: err

    PUSH_SUB_WITH_PROFILE(field_transfer_read_restart)

    call this%interpolation%read_restart(mesh, space, restart, err)

    POP_SUB_WITH_PROFILE(field_transfer_read_restart)
  end subroutine field_transfer_read_restart

  subroutine field_transfer_write_restart(this, mesh, space, restart, err)
    class(field_transfer_t), intent(inout) :: this
    class(mesh_t),           intent(in)    :: mesh
    class(space_t),          intent(in)    :: space
    type(restart_t),         intent(in)    :: restart
    integer,                 intent(out)   :: err

    PUSH_SUB_WITH_PROFILE(field_transfer_write_restart)

    call this%interpolation%write_restart(mesh, space, restart, err)

    POP_SUB_WITH_PROFILE(field_transfer_write_restart)
  end subroutine field_transfer_write_restart
end module field_transfer_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
