#include "global.h"

module atom_oct_m
  use debug_oct_m
  use global_oct_m
  use jellium_oct_m
  use messages_oct_m
  use profiling_oct_m
  use species_oct_m
  use unit_oct_m
  use unit_system_oct_m

  implicit none

  private
  public ::                               &
    atom_init,                            &
    atom_get_label,                       &
    atom_set_species,                     &
    atom_get_species,                     &
    atom_same_species,                    &
    all_species_are_jellium_slab,         &
    any_species_is_jellium_sphere

  type, public :: atom_t
    !private
    character(len=LABEL_LEN)  :: label = ""
    class(species_t), pointer  :: species => null() !< pointer to species
    integer, allocatable       :: c(:)      !< Constrain on te atom (0 or 1)

    !Components of the force
    real(real64), allocatable :: f_ii(:)      !< Ion-Ion part
    real(real64), allocatable :: f_vdw(:)     !< Van der Waals part
    real(real64), allocatable :: f_loc(:)     !< Local electronic part
    real(real64), allocatable :: f_nl(:)      !< NL electronic part
    real(real64), allocatable :: f_fields(:)  !< Lasers
    real(real64), allocatable :: f_u(:)       !< Hubbard forces
    real(real64), allocatable :: f_scf(:)     !< SCF forces
    real(real64), allocatable :: f_nlcc(:)    !< NLCC forces
    real(real64), allocatable :: f_photons(:) !< Photons forces
  contains
    procedure :: copy => atom_copy
    generic   :: assignment(=) => copy
    final :: atom_finalize
  end type atom_t

  interface atom_same_species
    module procedure atom_same_species_aa
    module procedure atom_same_species_as
  end interface atom_same_species

contains

  ! ---------------------------------------------------------
  subroutine atom_init(this, dim, label, species)
    type(atom_t),                      intent(out) :: this
    integer,                           intent(in)  :: dim
    character(len=*),                  intent(in)  :: label
    class(species_t), target, optional, intent(in)  :: species

    PUSH_SUB(atom_init)

    this%label = trim(adjustl(label))
    this%species => null()
    if (present(species)) this%species => species

    SAFE_ALLOCATE(this%c(1:dim))
    this%c = 0

    SAFE_ALLOCATE(this%f_ii(1:dim))
    SAFE_ALLOCATE(this%f_vdw(1:dim))
    SAFE_ALLOCATE(this%f_loc(1:dim))
    SAFE_ALLOCATE(this%f_nl(1:dim))
    SAFE_ALLOCATE(this%f_fields(1:dim))
    SAFE_ALLOCATE(this%f_u(1:dim))
    SAFE_ALLOCATE(this%f_scf(1:dim))
    SAFE_ALLOCATE(this%f_nlcc(1:dim))
    SAFE_ALLOCATE(this%f_photons(1:dim))
    this%f_ii      = M_ZERO
    this%f_vdw     = M_ZERO
    this%f_loc     = M_ZERO
    this%f_nl      = M_ZERO
    this%f_fields  = M_ZERO
    this%f_u       = M_ZERO
    this%f_scf     = M_ZERO
    this%f_nlcc    = M_ZERO
    this%f_photons = M_ZERO

    POP_SUB(atom_init)
  end subroutine atom_init

  ! ---------------------------------------------------------
  subroutine atom_copy(atom_out, atom_in)
    class(atom_t), intent(out) :: atom_out
    class(atom_t), intent(in)  :: atom_in

    PUSH_SUB(atom_copy)

    atom_out%label = atom_in%label
    atom_out%species => atom_in%species

    SAFE_ALLOCATE_SOURCE_A(atom_out%c,         atom_in%c)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_ii,      atom_in%f_ii)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_vdw,     atom_in%f_vdw)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_loc,     atom_in%f_loc)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_nl,      atom_in%f_nl)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_fields,  atom_in%f_fields)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_u,       atom_in%f_u)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_scf,     atom_in%f_scf)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_nlcc,    atom_in%f_nlcc)
    SAFE_ALLOCATE_SOURCE_A(atom_out%f_photons, atom_in%f_photons)

    POP_SUB(atom_copy)
  end subroutine atom_copy

  ! ---------------------------------------------------------
  impure elemental subroutine atom_finalize(this)
    type(atom_t), intent(inout) :: this

    PUSH_SUB(atom_finalize)

    this%label = ""
    this%species => null()

    SAFE_DEALLOCATE_A(this%c)

    SAFE_DEALLOCATE_A(this%f_ii)
    SAFE_DEALLOCATE_A(this%f_vdw)
    SAFE_DEALLOCATE_A(this%f_loc)
    SAFE_DEALLOCATE_A(this%f_nl)
    SAFE_DEALLOCATE_A(this%f_fields)
    SAFE_DEALLOCATE_A(this%f_u)
    SAFE_DEALLOCATE_A(this%f_scf)
    SAFE_DEALLOCATE_A(this%f_nlcc)
    SAFE_DEALLOCATE_A(this%f_photons)

    POP_SUB(atom_finalize)
  end subroutine atom_finalize

  ! ---------------------------------------------------------

  pure function atom_get_label(this) result(label)
    type(atom_t), intent(in) :: this

    character(len=len_trim(adjustl(this%label))) :: label

    label=trim(adjustl(this%label))

  end function atom_get_label

  ! ---------------------------------------------------------
  subroutine atom_set_species(this, species)
    type(atom_t),             intent(inout) :: this
    class(species_t), target, intent(in)    :: species

    PUSH_SUB(atom_set_species)

    this%species => species
    POP_SUB(atom_set_species)

  end subroutine atom_set_species

  ! ---------------------------------------------------------
  subroutine atom_get_species(this, species)
    type(atom_t),     target,  intent(in)  :: this
    class(species_t), pointer, intent(out) :: species

    ! NO PUSH_SUB, called too often

    species => null()
    if (associated(this%species)) species => this%species

  end subroutine atom_get_species

  ! ---------------------------------------------------------
  elemental function atom_same_species_aa(this, that) result(is)
    type(atom_t), intent(in) :: this
    type(atom_t), intent(in) :: that

    logical :: is

    is = (atom_get_label(this) == atom_get_label(that))

  end function atom_same_species_aa

  ! ---------------------------------------------------------
  elemental function atom_same_species_as(this, species) result(is)
    type(atom_t),     intent(in) :: this
    class(species_t), intent(in) :: species

    logical :: is

    is = (atom_get_label(this) == species%get_label())

  end function atom_same_species_as

  !> @brief Check if all species are jellium slab
  pure logical function all_species_are_jellium_slab(atom)
    type(atom_t), intent(in)  :: atom(:)

    integer                   :: i

    all_species_are_jellium_slab = .true.
    do i = 1, size(atom)
      select type(spec => atom(i)%species)
      type is(jellium_slab_t)
      class default
        all_species_are_jellium_slab = .false.
      end select
    enddo

  end function all_species_are_jellium_slab

  !> @brief Check if any species is a jellium sphere
  pure logical function any_species_is_jellium_sphere(atom)
    type(atom_t), intent(in)  :: atom(:)

    integer                   :: i

    any_species_is_jellium_sphere = .false.
    do i = 1, size(atom)
      select type(spec => atom(i)%species)
      type is(jellium_sphere_t)
        any_species_is_jellium_sphere = .true.
      end select
    enddo

  end function any_species_is_jellium_sphere

end module atom_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
