!! Copyright (C) 2019 R. Jestaedt, H. Appel, F. Bonafe, M. Oliveira, N. Tancogne-Dejean
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

!>--------------------------------------------------------------
!! This module defines "Maxwell functions", to be used by
!! the Maxwell module especially for Maxwell incident waves
!!--------------------------------------------------------------
module maxwell_function_oct_m
  use iso_c_binding
  use debug_oct_m
  use fft_oct_m
  use global_oct_m
  use io_oct_m
  use loct_math_oct_m
  use math_oct_m
  use mpi_oct_m
  use parser_oct_m
  use profiling_oct_m
  use splines_oct_m
  use string_oct_m
  use unit_oct_m
  use unit_system_oct_m
  use namespace_oct_m

  implicit none

  public ::                      &
    mxf_t,                       &
    mxf_init,                    &
    mxf_init_const_wave,         &
    mxf_init_const_phase,        &
    mxf_init_gaussian_wave,      &
    mxf_init_cosinoidal_wave,    &
    mxf_init_fromexpr,           &
    mxf,                         &
    mxf_read,                    &
    mxf_is_empty

  integer, public, parameter ::     &
    MXF_EMPTY            =  10001,  &
    MXF_CONST_WAVE       =  10002,  &
    MXF_CONST_PHASE      =  10004,  &
    MXF_GAUSSIAN_WAVE    =  10005,  &
    MXF_COSINOIDAL_WAVE  =  10006,  &
    MXF_LOGISTIC_WAVE    =  10007,  &
    MXF_TRAPEZOIDAL_WAVE =  10008,  &
    MXF_FROM_FILE        =  10009,  &
    MXF_NUMERICAL        =  10010,  &
    MXF_FROM_EXPR        =  10011,  &
    MXF_FOURIER_SERIES   =  10012,  &
    MXF_ZERO_FOURIER     =  10013

  type mxf_t
    integer :: mode              = MXF_EMPTY
    real(real64)   :: k_vector(3) = M_ZERO
    real(real64)   :: r0(3)       = M_ZERO  !< vector at the maximum of the pulse
    real(real64)   :: width             = M_ZERO  !< the width of the pulse
    real(real64)   :: a0                = M_ZERO
    real(real64)   :: dx                = M_ZERO !< the space-discretization value.
    real(real64)   :: init_x            = M_ZERO
    real(real64)   :: final_x           = M_ZERO
    real(real64)   :: growth            = M_ZERO
    integer :: niter             = 0
    integer :: nfreqs            = 0

    type(spline_t)         :: amplitude
    character(len=200)     :: expression
    type(fft_t) :: fft_handler
  end type mxf_t

  interface mxf
    module procedure mxf_eval
  end interface mxf

contains

  !------------------------------------------------------------
  !> This function initializes "f" from the MXFunctions block.
  subroutine mxf_read(f, namespace, function_name, ierr)
    type(mxf_t),       intent(inout) :: f
    type(namespace_t), intent(in)    :: namespace
    character(len=*),  intent(in)    :: function_name
    integer,           intent(out)   :: ierr  !< Error code, 0 on success.

    type(block_t) :: blk
    integer :: nrows, ncols, i, function_type, idim
    character(len=100) :: row_name, function_expression
    real(real64) :: a0, r0(3), growth, width, k_vector(3)

    PUSH_SUB(mxf_read)

    !%Variable MaxwellFunctions
    !%Type block
    !%Section Maxwell
    !%Description
    !% This block specifies the shape of a "spatial-dependent function", such as the
    !% envelope needed when using the <tt>MaxwellFunctions</tt> block. Each line in the block
    !% specifies one function. The first element of each line will be a string
    !% that defines the name of the function. The second element specifies which type
    !% of function we are using; in the following we provide an example for each of the
    !% possible types:
    !%
    !%Option mxf_const_wave 10002
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_const_wave | kx | ky | kz | x0 | y0 | z0
    !% <br>%</tt>
    !%
    !% The function is constant plane wave <math> f(x,y,z) = a0 * \cos( kx*(x-x0) + ky*(y-y0) + kz*(z-z0) ) </math>
    !%
    !%Option mxf_const_phase 10004
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_const_phase | kx | ky | kz | x0 | y0 | z0
    !% <br>%</tt>
    !%
    !% The function is a constant phase of <math> f(x,y,z) = a0 * (kx * x0 + ky * y0 + kz * z0) </math>
    !%
    !%Option mxf_gaussian_wave 10005
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_gaussian_wave | kx | ky | kz | x0 | y0 | z0 | width
    !% <br>%</tt>
    !%
    !% The function is a Gaussian, <math> f(x,y,z) = a0 * \exp( -( kx*(x-x0) + ky*(y-y0) + kz*(z-z0) )^2 / (2 width^2) ) </math>
    !%
    !%Option mxf_cosinoidal_wave 10006
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_cosinoidal_wave | kx | ky | kz | x0 | y0 | z0 | width
    !% <br>%</tt>
    !%
    !% <math> f(x,y,z) =  \cos( \frac{\pi}{2} \frac{kx*(x-x0)+ky*(y-y0)+kz*(z-z0)-2 width}{width} + \pi )  </math>
    !%
    !% If <math> | kx*x + ky*y + kz*z - x0 | > \xi\_0 </math>, then <math> f(x,y,z) = 0 </math>.
    !%
    !%Option mxf_logistic_wave 10007
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_logistic_wave | kx | ky | kz | x0 | y0 | z0 | growth | width
    !% <br>%</tt>
    !%
    !% The function is a logistic function, <math> f(x,y,z) = a0 * 1/(1+\exp(growth*(kx*(x-x0)+ky*(y-y0)+kz*(kz*(z-z0))+width/2))) * 1/(1+\exp(-growth*(kx*(x-x0)+ky*(y-y0)+kz*(kz*(z-z0))-width/2)))  </math>
    !%
    !%Option mxf_trapezoidal_wave 10008
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_trapezoidal_wave | kx | ky | kz | x0 | y0 | z0 | growth | width
    !% <br>%</tt>
    !%
    !% The function is a logistic function,
    !%   <br> <math> f(x,y,z)      = a0 * ( ( 1-growth*(k*(r-r0)-width/2)*\Theta(k*(r-r0)-width/2))*\Theta(-(k*(r-r0)+width/2+1/growth)) </math>
    !%   <br> <math> \qquad \qquad \qquad + (-1+growth*(k*(r-r0)+width/2)*\Theta(k*(r-r0)+width/2))*\Theta(-(k*(r-r0)-width/2+1/growth)) ) </math>
    !%
    !%Option mxf_from_expr 10011
    !%
    !% <tt>%MaxwellFunctions
    !% <br>&nbsp;&nbsp; "function-name" | mxf_from_expr | kx | ky | kz | "expression"
    !% <br>%</tt>
    !%
    !% The temporal shape of the field is given as an expression (e.g., <tt>cos(2.0*x-3*y+4*z)</tt>. The
    !% letter <i>x</i>, <i>y</i>, <i>z</i> means spatial coordinates, obviously.
    !% The expression is used to construct the function <i>f</i>
    !% that defines the field.
    !%End
    ierr = -3
    if (parse_block(namespace, 'MaxwellFunctions', blk) /= 0) then
      ierr = -1
      POP_SUB(mxf_read)
      return
    end if

    nrows = parse_block_n(blk)
    row_loop: do i = 1, nrows
      call parse_block_string(blk, i-1, 0, row_name)
      if (trim(row_name) == trim(function_name)) then

        ncols = parse_block_cols(blk, i-1)
        call parse_block_integer(blk, i-1, 1, function_type)

        a0 = M_ONE
        r0 = M_ZERO
        width = M_ZERO
        k_vector = M_ZERO
        select case (function_type)
        case (MXF_CONST_WAVE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call mxf_init_const_wave(f, a0, k_vector, r0)
        case (MXF_CONST_PHASE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call mxf_init_const_phase(f, a0, k_vector, r0)
        case (MXF_GAUSSIAN_WAVE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call parse_block_float(blk, i-1, 8, width, units_inp%length)
          call mxf_init_gaussian_wave(f, a0, k_vector, r0, width)
        case (MXF_COSINOIDAL_WAVE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call parse_block_float(blk, i-1, 8, width, units_inp%length)
          call mxf_init_cosinoidal_wave(f, a0, k_vector, r0, width)
        case (MXF_LOGISTIC_WAVE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call parse_block_float(blk, i-1, 8, growth, units_inp%length)
          call parse_block_float(blk, i-1, 9, width, units_inp%length)
          call mxf_init_logistic_wave(f, a0, k_vector, r0, growth, width)
        case (MXF_TRAPEZOIDAL_WAVE)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          do idim = 1, 3
            call parse_block_float(blk, i-1, 4+idim, r0(idim), units_inp%length)
          end do
          call parse_block_float(blk, i-1, 8, growth, units_inp%length)
          call parse_block_float(blk, i-1, 9, width, units_inp%length)
          call mxf_init_trapezoidal_wave(f, a0, k_vector, r0, growth, width)
        case (MXF_FROM_EXPR)
          do idim = 1, 3
            call parse_block_float(blk, i-1, 1+idim, k_vector(idim), unit_one/units_inp%length)
          end do
          call parse_block_string(blk, i-1, 5, function_expression)
          call conv_to_C_string(function_expression)
          call mxf_init_fromexpr(f, k_vector, trim(function_expression))
        case default
          ierr = -2
          call parse_block_end(blk)
          POP_SUB(mxf_read)
          return
        end select

        ierr = 0
        exit row_loop
      end if
    end do row_loop

    call parse_block_end(blk)

    POP_SUB(mxf_read)
  end subroutine mxf_read
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init(f)
    type(mxf_t), intent(inout) :: f

    PUSH_SUB(mxf_init)

    f%mode = MXF_EMPTY
    f%niter = 0
    f%dx = M_ZERO

    POP_SUB(mxf_init)
  end subroutine mxf_init
  !------------------------------------------------------------


  !------------------------------------------------------------
  logical function mxf_is_empty(f)
    type(mxf_t), intent(in) :: f

    PUSH_SUB(mxf_is_empty)
    mxf_is_empty = (f%mode == MXF_EMPTY)

    POP_SUB(mxf_is_empty)
  end function mxf_is_empty
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_const_wave(f, a0, k_vector, r0)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3)

    PUSH_SUB(mxf_init_const_wave)

    f%mode = MXF_CONST_WAVE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0

    POP_SUB(mxf_init_const_wave)
  end subroutine mxf_init_const_wave
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_const_phase(f, a0, k_vector, r0)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3)

    PUSH_SUB(mxf_init_const_phase)

    f%mode = MXF_CONST_PHASE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0

    POP_SUB(mxf_init_const_phase)
  end subroutine mxf_init_const_phase
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_gaussian_wave(f, a0, k_vector, r0, width)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3), width

    PUSH_SUB(mxf_init_gaussian_wave)

    f%mode = MXF_GAUSSIAN_WAVE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0
    f%width = width

    POP_SUB(mxf_init_gaussian_wave)
  end subroutine mxf_init_gaussian_wave
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_cosinoidal_wave(f, a0, k_vector, r0, width)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3), width

    PUSH_SUB(mxf_init_cosinoidal_wave)

    f%mode = MXF_COSINOIDAL_WAVE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0
    f%width = width

    POP_SUB(mxf_init_cosinoidal_wave)
  end subroutine mxf_init_cosinoidal_wave
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_logistic_wave(f, a0, k_vector, r0, growth, width)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3), growth, width

    PUSH_SUB(mxf_init_logistic_wave)

    f%mode = MXF_LOGISTIC_WAVE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0
    f%growth = growth
    f%width = width

    POP_SUB(mxf_init_logistic_wave)
  end subroutine
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_trapezoidal_wave(f, a0, k_vector, r0, growth, width)
    type(mxf_t), intent(inout) :: f
    real(real64),       intent(in)    :: a0, k_vector(3), r0(3), growth, width

    PUSH_SUB(mxf_init_trapezoidal_wave)

    f%mode = MXF_TRAPEZOIDAL_WAVE
    f%a0 = a0
    f%k_vector = k_vector
    f%r0 = r0
    f%growth = growth
    f%width = width

    POP_SUB(mxf_init_trapezoidal_wave)
  end subroutine
  !------------------------------------------------------------


  !------------------------------------------------------------
  subroutine mxf_init_fromexpr(f, k_vector, expression)
    type(mxf_t),      intent(inout) :: f
    real(real64),      intent(in)   :: k_vector(3)
    character(len=*), intent(in)    :: expression

    PUSH_SUB(mxf_init_fromexpr)

    f%mode       = MXF_FROM_EXPR
    f%k_vector   = k_vector
    f%expression = trim(expression)

    POP_SUB(mxf_init_fromexpr)
  end subroutine mxf_init_fromexpr
  !------------------------------------------------------------

  !> @brief Evaluation of envelope itself.
  !!
  !! This part uses the "spatial-dependent function" defined in <tt>MaxwellFunctions</tt>
  !! block. It calculates the value of the envelope solely, and feeds it to mxf_eval .
  !------------------------------------------------------------
  complex(real64) function mxf_envelope_eval(f, x) result(env)
    type(mxf_t), intent(in) :: f         !< spatial envelope function
    real(real64),       intent(in) :: x(:)      !< position
    real(real64) :: xx, limit_1, limit_2, limit_3, limit_4, r_re, r_im, rr
    complex(real64) :: r
    integer :: xdim

    xdim = size(x)

    select case (f%mode)
    case (MXF_CONST_WAVE)

      env = f%a0

    case (MXF_CONST_PHASE)

      env = f%a0 * sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim)))

    case (MXF_GAUSSIAN_WAVE)

      r = exp(-((sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) &
        / norm2(f%k_vector(1:xdim)) )**2 / (M_TWO*f%width**2)))
      env = f%a0 * r

    case (MXF_COSINOIDAL_WAVE)

      r = M_ZERO
      if (abs( sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))/norm2(f%k_vector(1:xdim)))) <= f%width) then
        r = - cos((M_Pi/M_TWO) * ((sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) &
          / norm2(f%k_vector(1:xdim)) - M_TWO*f%width) / f%width))
      end if
      env = f%a0 * r

    case (MXF_LOGISTIC_WAVE)

      r = M_ONE/(M_ONE + exp(f%growth*(sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) &
        / norm2(f%k_vector(1:xdim)) - f%width/M_TWO))) &
        + M_ONE/(M_ONE + exp(-f%growth*(sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) &
        / norm2(f%k_vector(1:xdim)) + f%width/M_TWO))) - M_ONE
      env = f%a0 * r

    case (MXF_TRAPEZOIDAL_WAVE)

      xx = sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) / norm2(f%k_vector(1:xdim))
      limit_1 = - f%width/M_TWO - M_ONE/f%growth
      limit_2 = - f%width/M_TWO
      limit_3 =   f%width/M_TWO
      limit_4 =   f%width/M_TWO + M_ONE/f%growth
      if ((xx > limit_1) .and. (xx <= limit_2)) then
        r = M_ONE + f%growth * (sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim)))/norm2(f%k_vector(1:xdim)) + f%width/M_TWO)
      else if ((xx > limit_2) .and. (xx <= limit_3)) then
        r = M_ONE
      else if ((xx > limit_3) .and. (xx <= limit_4)) then
        r = M_ONE - f%growth * (sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim)))/norm2(f%k_vector(1:xdim)) - f%width/M_TWO)
      else
        r = M_ZERO
      end if
      env = f%a0 * r

    case (MXF_FROM_EXPR)
      ! here x(:) is actually x(:) - v(:)*t
      rr = norm2(x)
      call parse_expression(r_re, r_im, 3, x(:), rr, M_ZERO, trim(f%expression))
      env = cmplx(r_re, r_im, real64)

    case default

      env = M_ZERO

    end select

  end function mxf_envelope_eval
  !------------------------------------------------------------

  !> @brief Evaluation of spatial envelope Functions.
  !!
  !! This part uses the "spatial-dependent function" defined in <tt>MaxwellFunctions</tt>
  !! block. This function is evaluated on the grid point and the 'spatially-enveloped' value of the field is returned.
  !! In plane wave, spatial part is evaluated in mxf_eval, while for other forms solely envelope
  !! is evaluated
  !------------------------------------------------------------
  complex(real64) function mxf_eval(f, x, phi) result(y)
    type(mxf_t), intent(in) :: f         !< spatial envelope function
    real(real64),       intent(in) :: x(:)      !< position
    real(real64), optional, intent(in) :: phi   !< phase, optional

    real(real64) :: phi_
    integer :: xdim

    ! no push_sub because it is called too frequently

    phi_ = optional_default(phi, M_ZERO)

    xdim = size(x)

    y = mxf_envelope_eval(f, x)
    if (f%mode /= MXF_CONST_PHASE) then
      y = y * exp(M_zI * (sum(f%k_vector(1:xdim)*(x(:) - f%r0(1:xdim))) + phi_))
    endif

  end function mxf_eval
  !------------------------------------------------------------

end module maxwell_function_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
