!! Copyright (C) 2020 M. Oliveira
!!
!! This Source Code Form is subject to the terms of the Mozilla Public
!! License, v. 2.0. If a copy of the MPL was not distributed with this
!! file, You can obtain one at https://mozilla.org/MPL/2.0/.
!!

#include "global.h"

!> @brief This module defines classes and functions for interaction partners.
!!
!! Interaction partners are general objects, which define the "source" of an interaction (interaction_oct_m::interaction_t).
module interaction_partner_oct_m
  use debug_oct_m
  use global_oct_m
  use interaction_surrogate_oct_m
  use iteration_counter_oct_m
  use linked_list_oct_m
  use messages_oct_m
  use multisystem_debug_oct_m
  use namespace_oct_m
  use quantity_oct_m
  implicit none

  private
  public ::                &
    interaction_partner_t, &
    partner_list_t,        &
    partner_iterator_t

  integer, parameter, public ::   &
    COUPLINGS_UNDEFINED      = 0, &
    COUPLINGS_BEHIND_IN_TIME = 1, &
    COUPLINGS_ON_TIME        = 2, &
    COUPLINGS_AHEAD_IN_TIME  = 3

  !> @brief abstract class for general interaction partners
  !!
  !! Some interactions require a partner. This is usually a system, but it could
  !! also be some external entity, like an external field.
  !! An interaction partner must expose some quantities that the interaction can use.
  type, abstract :: interaction_partner_t
    private
    type(namespace_t), public :: namespace

    integer, allocatable, public :: supported_interactions_as_partner(:) !< list of interactions, which support
    !!                                                                      this interaction_partner_t as partner

    type(quantity_list_t), public :: quantities       !< List of quantities known by the system.
  contains
    procedure :: add_partners_to_list => interaction_partner_add_partners_to_list
    procedure :: update_quantity  => interaction_partner_update_quantity !< @copydoc interaction_partner_update_quantity
    procedure :: update_on_demand_quantity => interaction_partner_update_on_demand_quantity !< @copydoc interaction_partner_update_on_demand_quantity
    procedure :: update_on_demand_quantities  => interaction_partner_update_on_demand_quantities !< @copydoc interaction_partner_update_on_demand_quantities
    procedure :: check_couplings_status  => interaction_partner_check_couplings_status !< @copydoc interaction_partner_check_couplings_status
    procedure(interaction_partner_init_interaction_as_partner),    deferred :: init_interaction_as_partner
    !< @copydoc interaction_partner_init_interaction_as_partner
    procedure(interaction_partner_copy_quantities_to_interaction), deferred :: copy_quantities_to_interaction
    !< @copydoc interaction_partner_copy_quantities_to_interaction
  end type interaction_partner_t

  abstract interface
    ! ---------------------------------------------------------
    subroutine interaction_partner_init_interaction_as_partner(partner, interaction)
      import interaction_partner_t
      import interaction_surrogate_t
      class(interaction_partner_t),     intent(in)    :: partner     !< the current interacion partner (this)
      class(interaction_surrogate_t),   intent(inout) :: interaction !< the interaction
    end subroutine interaction_partner_init_interaction_as_partner

    ! ---------------------------------------------------------
    subroutine interaction_partner_copy_quantities_to_interaction(partner, interaction)
      import interaction_partner_t
      import interaction_surrogate_t
      class(interaction_partner_t),     intent(inout) :: partner     !< the current interacion partner (this)
      class(interaction_surrogate_t),   intent(inout) :: interaction !< the interaction
    end subroutine interaction_partner_copy_quantities_to_interaction

  end interface

  !> @brief the list of partners
  !!
  !! This class extends the list to create a partner list
  type, extends(linked_list_t) :: partner_list_t
    private
  contains
    procedure :: add => partner_list_add_node !< @copydoc interaction_partner_oct_m::partner_list_add_node
  end type partner_list_t

  !> @brief iterator for the list of partners
  !!
  !! This class extends the list iterator to create a partner list iterator
  type, extends(linked_list_iterator_t) :: partner_iterator_t
    private
  contains
    procedure :: get_next => partner_iterator_get_next !< @copydoc interaction_partner_oct_m::partner_iterator_get_next
  end type partner_iterator_t

contains

  ! ---------------------------------------------------------
  !> @brief add interaction partner to a list
  !!
  !! If the interaciton_type is provided, the partner is only added to the list
  !! if it supports that interaction.
  recursive subroutine interaction_partner_add_partners_to_list(this, list, interaction_type)
    class(interaction_partner_t),           intent(in)    :: this
    class(partner_list_t),                  intent(inout) :: list
    integer,                      optional, intent(in)    :: interaction_type

    if (present(interaction_type)) then
      if (any(this%supported_interactions_as_partner == interaction_type)) then
        call list%add(this)
      end if
    else
      call list%add(this)
    end if

  end subroutine interaction_partner_add_partners_to_list

  ! ---------------------------------------------------------
  !> @brief Method to be overriden by interaction partners
  !! that have quantities that can be updated on demand.
  !!
  !! This routine simply throws an error, as it is not mean to be called. We
  !! could have implemented this as a deferred method, but we prefer not to
  !! force interaction partners that do not have quantities to be updated on
  !! demand to implement it.
  subroutine interaction_partner_update_quantity(this, label)
    class(interaction_partner_t), intent(inout) :: this
    character(len=*),             intent(in)    :: label

    PUSH_SUB(interaction_partner_update_quantity)

    write(message(1), '(a,a,a,a,a)') 'Interation partner "', trim(this%namespace%get()), &
      '"does not know how to update quantity"', trim(label), '".'
    call messages_fatal(1, namespace=this%namespace)

    POP_SUB(interaction_partner_update_quantity)
  end subroutine interaction_partner_update_quantity

  !! NOTE: here we update the quantity to be exactly at
  !! requested_iteration. This is different than what is done in
  !! interaction_partner_update_on_demand_quantities and it means the caller
  !! needs to pass the correct requested_iteration.
  recursive subroutine interaction_partner_update_on_demand_quantity(this, quantity, requested_iteration)
    class(interaction_partner_t), intent(inout) :: this

    type(quantity_t),             intent(inout) :: quantity
    class(iteration_counter_t),   intent(in)    :: requested_iteration

    integer :: i
    type(quantity_t), pointer :: parent

    ! Before updating this quantity, we also need to update its parents
    do i = 1, size(quantity%parents)
      parent => this%quantities%get(quantity%parents(i))

      ! We are only allowed to update on demand quantities that are behind the requested iteration
      if (parent%iteration >= requested_iteration .or. .not. parent%updated_on_demand) cycle

      call this%update_on_demand_quantity(parent, requested_iteration)
    end do

    ! If all parents are at the requested time, we can proceed with updating the quantity itself
    ! Note that here we ignore quantities that are available at any time.
    if (all(this%quantities%iteration_equal(quantity%parents, requested_iteration) .or. &
      this%quantities%always_available(quantity%parents))) then

      quantity%iteration = requested_iteration
      call multisystem_debug_write_marker(this%namespace, event_iteration_update_t("quantity", quantity%label, &
        quantity%iteration, "set"))
      call this%update_quantity(quantity%label)
    end if

  end subroutine interaction_partner_update_on_demand_quantity


  ! ---------------------------------------------------------
  !> @brief Given a list of quantities, update the ones that can be update on demand
  !!
  !! For each quantity that is not always available, the update will only take
  !! place if the quantity can be evaluated at the requested
  !! iteration. Currently we also allow for a special case: some quantities are
  !! allowed to get ahead by one iteration if "retardation_allowed" is set to
  !! true.
  subroutine interaction_partner_update_on_demand_quantities(this, labels, requested_iteration, retardation_allowed)
    class(interaction_partner_t), target, intent(inout) :: this
    character(len=*),                     intent(in)    :: labels(:)
    class(iteration_counter_t),           intent(in)    :: requested_iteration
    logical,                              intent(in)    :: retardation_allowed

    integer :: iq
    type(quantity_t), pointer :: quantity

    do iq = 1, size(labels)
      ! Get a pointer to the quantity
      quantity => this%quantities%get(labels(iq))

      ! We are only updating on demand quantities that are behind the requested iteration
      if (quantity%iteration >= requested_iteration .or. .not. quantity%updated_on_demand) cycle

      if (quantity%always_available) then
        ! We set the quantity iteration to the requested one, so that the partner
        ! can use this information when updating the quantity
        call this%update_on_demand_quantity(quantity, requested_iteration)

      else if (quantity%iteration + 1 <= requested_iteration .or. &
        (retardation_allowed .and. quantity%iteration + 1 > requested_iteration)) then
        ! We can update because the partner will reach this iteration in the next algorithmic step
        ! For retarded interactions, we need to allow the quantity to get ahead by one iteration
        call this%update_on_demand_quantity(quantity, quantity%iteration + 1)

      end if
    end do

  end subroutine interaction_partner_update_on_demand_quantities

  ! ---------------------------------------------------------
  !> Check the status of some couplings.
  !!
  !! Possible results are:
  !!  - COUPLINGS_UNDEFINED:      if there are some couplings ahead in time and some on time
  !!  - COUPLINGS_BEHIND_IN_TIME: if any coupling is behind in time and COUPLINGS_UNDEFINED condition is not met
  !!  - COUPLINGS_ON_TIME:        if all couplings are right on time
  !!  - COUPLINGS_AHEAD_IN_TIME:  if all couplings are ahead in time
  integer function interaction_partner_check_couplings_status(this, couplings, requested_iteration) result(status)
    class(interaction_partner_t), intent(inout) :: this
    character(len=*),             intent(in)    :: couplings(:)
    class(iteration_counter_t),   intent(in)    :: requested_iteration

    type(quantity_t), pointer :: coupling
    integer :: i, ahead, on_time, relevant_couplings
    character(len=200) :: marker_info
    character(len=20) :: status_string
    type(event_handle_t) :: debug_handle


    PUSH_SUB(interaction_partner_check_couplings_status)

    debug_handle = multisystem_debug_write_event_in(system_namespace = this%namespace, &
      event = event_function_call_t("check_couplings_status"), &
      requested_iteration = requested_iteration)


    ! Count couplings on time and ahead
    on_time = 0
    ahead = 0
    relevant_couplings = 0
    do i = 1, size(couplings)
      coupling => this%quantities%get(couplings(i))

      ! Couplings that are available at any time do not affect the status, so we will ignore them
      if (coupling%always_available) cycle

      relevant_couplings = relevant_couplings + 1
      if (coupling%iteration == requested_iteration) on_time = on_time + 1
      if (coupling%iteration > requested_iteration) ahead = ahead + 1
    end do

    ! Determine status
    if (on_time > 0 .and. ahead > 0) then
      status = COUPLINGS_UNDEFINED
      status_string = "UNDEFINED"
    else if (on_time + ahead < relevant_couplings) then
      status = COUPLINGS_BEHIND_IN_TIME
      status_string = "BEHIND"
    else if (on_time == relevant_couplings) then
      status = COUPLINGS_ON_TIME
      status_string = "ON_TIME"
    else if (ahead == relevant_couplings) then
      status = COUPLINGS_AHEAD_IN_TIME
      status_string = "AHEAD"
    end if

    write(marker_info, '(A20," check_couplings_status: ahead = ",I5,", on_time = ",I5,", relevant = ",I5, ", status = ",A9)') &
      trim(this%namespace%get()), ahead, on_time, relevant_couplings, trim(status_string)
    call multisystem_debug_write_event_out(debug_handle, extra=marker_info, requested_iteration=requested_iteration)

    POP_SUB(interaction_partner_check_couplings_status)
  end function interaction_partner_check_couplings_status

  ! ---------------------------------------------------------
  !> @brief add a partner to the list
  !!
  subroutine partner_list_add_node(this, partner)
    class(partner_list_t)                :: this     !< the partner list
    class(interaction_partner_t), target :: partner  !< the partner to add

    PUSH_SUB(partner_list_add_node)

    call this%add_ptr(partner)

    POP_SUB(partner_list_add_node)
  end subroutine partner_list_add_node

  ! ---------------------------------------------------------
  !> @brief get next partner from the list
  !!
  function partner_iterator_get_next(this) result(partner)
    class(partner_iterator_t),    intent(inout) :: this     !< the partner list
    class(interaction_partner_t), pointer       :: partner  !< the next element of the list

    PUSH_SUB(partner_iterator_get_next)

    select type (ptr => this%get_next_ptr())
    class is (interaction_partner_t)
      partner => ptr
    class default
      ASSERT(.false.)
    end select

    POP_SUB(partner_iterator_get_next)
  end function partner_iterator_get_next

end module interaction_partner_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
