!! Copyright (C) 2020 Nicolas Tancogne-Dejean, Sebastian Ohlmann, Heiko Appel
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module propagator_exp_mid_2step_oct_m
  use, intrinsic :: iso_fortran_env
  use algorithm_oct_m
  use debug_oct_m
  use global_oct_m
  use propagator_oct_m

  implicit none

  private
  public ::                            &
    propagator_exp_mid_2step_t

  !> Implements the implicit exponential midpoint propagator with predictor-corrector

  type, extends(propagator_t) :: propagator_exp_mid_2step_t
    private
  end type propagator_exp_mid_2step_t

  interface propagator_exp_mid_2step_t
    procedure propagator_exp_mid_2step_constructor
  end interface propagator_exp_mid_2step_t

  !# doc_start exp_mid_2step_propagation_operations
  ! Specific exponential mid-point propagation operations identifiers
  character(len=ALGO_LABEL_LEN), public, parameter :: &
    EXPMID_2STEP_START        = 'EXPMID_2STEP_START',             &
    EXPMID_2STEP_FINISH       = 'EXPMID_2STEP_FINISH',            &
    EXPMID_2STEP_PREDICT_DT_2 = 'EXPMID_2STEP_PREDICT_DT_2',      &
    EXPMID_2STEP_PREDICT_DT   = 'EXPMID_2STEP_PREDICT_DT',        &
    EXPMID_2STEP_CORRECT_DT_2 = 'EXPMID_2STEP_CORRECT_DT_2',      &
    UPDATE_HAMILTONIAN  = 'UPDATE_HAMILTONIAN'

  ! Specific exponential mid-point propagation operations
  type(algorithmic_operation_t), public, parameter :: &
    OP_EXPMID_2STEP_START        = algorithmic_operation_t(EXPMID_2STEP_START,        &
    'Starting exponential mid-point propagation'),  &
    OP_EXPMID_2STEP_FINISH       = algorithmic_operation_t(EXPMID_2STEP_FINISH,       &
    'Finishing exponential mid-point propagation'), &
    OP_EXPMID_2STEP_PREDICT_DT_2 = algorithmic_operation_t(EXPMID_2STEP_PREDICT_DT_2, &
    'Prediction step - Predicting state at dt/2 '), &
    OP_EXPMID_2STEP_PREDICT_DT   = algorithmic_operation_t(EXPMID_2STEP_PREDICT_DT,   &
    'Prediction step - Predicting state at dt'),    &
    OP_EXPMID_2STEP_CORRECT_DT_2 = algorithmic_operation_t(EXPMID_2STEP_CORRECT_DT_2, &
    'Correction step - Correcting state at dt/2'),  &
    OP_UPDATE_HAMILTONIAN  = algorithmic_operation_t(UPDATE_HAMILTONIAN,  'Updating Hamiltonian')
  !# doc_end

contains

  ! ---------------------------------------------------------
  function propagator_exp_mid_2step_constructor(dt, predictor_corrector) result(this)
    real(real64),              intent(in) :: dt
    logical,                   intent(in) :: predictor_corrector
    type(propagator_exp_mid_2step_t), pointer   :: this

    PUSH_SUB(propagator_exp_mid_2step_constructor)

    allocate(this)

    this%predictor_corrector = predictor_corrector
    this%start_operation = OP_EXPMID_2STEP_START
    this%final_operation = OP_EXPMID_2STEP_FINISH

    if (predictor_corrector) then

      call this%add_operation(OP_STORE_CURRENT_STATUS)
      call this%add_operation(OP_EXPMID_2STEP_PREDICT_DT_2)  ! predict: psi(t+dt/2) = 0.5*(U_H(dt) psi(t) + psi(t)) or via extrapolation
      call this%add_operation(OP_UPDATE_COUPLINGS)
      call this%add_operation(OP_START_SCF_LOOP)
      call this%add_operation(OP_UPDATE_INTERACTIONS)
      call this%add_operation(OP_UPDATE_HAMILTONIAN)         ! update: H(t+dt/2) from psi(t+dt/2)
      call this%add_operation(OP_EXPMID_2STEP_PREDICT_DT)    ! predict: psi(t+dt) = U_H(t+dt/2) psi(t)
      call this%add_operation(OP_EXPMID_2STEP_CORRECT_DT_2)  ! correct: psi(t+dt/2) = 0.5*(psi(t+dt) + psi(t))
      call this%add_operation(OP_END_SCF_LOOP)
      call this%add_operation(OP_ITERATION_DONE)
      call this%add_operation(OP_REWIND_ALGORITHM)

      this%max_scf_count = 10
      this%scf_tol = 1e-6_real64 ! At the moment arbitrary. This is system specific and should be adapted.

    else

      call this%add_operation(OP_STORE_CURRENT_STATUS)
      call this%add_operation(OP_EXPMID_2STEP_PREDICT_DT_2)  ! predict: psi(t+dt/2) = 0.5*(U_H(dt) psi(t) + psi(t)) or via extrapolation
      call this%add_operation(OP_UPDATE_COUPLINGS)
      call this%add_operation(OP_UPDATE_INTERACTIONS)
      call this%add_operation(OP_UPDATE_HAMILTONIAN)         ! update: H(t+dt/2) from psi(t+dt/2)
      call this%add_operation(OP_EXPMID_2STEP_PREDICT_DT)    ! predict: psi(t+dt) = U_H(t+dt/2) psi(t)
      call this%add_operation(OP_UPDATE_COUPLINGS)
      call this%add_operation(OP_UPDATE_INTERACTIONS)
      call this%add_operation(OP_UPDATE_HAMILTONIAN)         ! update: H(t+dt) from psi(t+dt)
      call this%add_operation(OP_ITERATION_DONE)
      call this%add_operation(OP_REWIND_ALGORITHM)

    end if

    ! The implicit exponential midpoint has two algorithmic steps
    this%algo_steps = 2

    this%dt = dt

    POP_SUB(propagator_exp_mid_2step_constructor)
  end function propagator_exp_mid_2step_constructor

end module propagator_exp_mid_2step_oct_m


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
