!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch.
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

 ! ---------------------------------------------------------
subroutine output_hamiltonian(outp, namespace, space, dir, hm, st, der, ions, gr, iter, grp)
  type(output_t),            intent(in)    :: outp
  type(namespace_t),         intent(in)    :: namespace
  class(space_t),            intent(in)    :: space
  character(len=*),          intent(in)    :: dir
  type(hamiltonian_elec_t),  intent(in)    :: hm
  type(states_elec_t),       intent(inout) :: st
  type(derivatives_t),       intent(in)    :: der
  type(ions_t),              intent(in)    :: ions
  type(grid_t),              intent(in)    :: gr
  integer,                   intent(in)    :: iter
  type(mpi_grp_t), optional, intent(in)    :: grp !< the group that shares the same data, must contain the domains group

  integer :: is, err, idir, ik, ib, ist
  character(len=MAX_PATH_LEN) :: fname
  real(real64), allocatable :: vh(:), v0(:,:), nxc(:), potential(:)
  real(real64), allocatable :: current_kpt(:, :)
  real(real64), allocatable :: density_kpt(:, :), density_tmp(:,:)
  type(density_calc_t) :: dens_calc
  real(real64), allocatable :: gradvh(:, :), heat_current(:, :, :)

  PUSH_SUB(output_hamiltonian)


  if (outp%what_now(OPTION__OUTPUT__POTENTIAL, iter)) then
    call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, "v0", namespace, &
      space, der%mesh, hm%ep%vpsl, units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)

    if (allocated(hm%v_static)) then
      call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, "vext", namespace, &
        space, der%mesh, hm%v_static, units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
    end if

    if (hm%theory_level /= INDEPENDENT_PARTICLES) then

      call hm%ks_pot%output_potentials(namespace, outp%how(OPTION__OUTPUT__POTENTIAL), dir, space, der%mesh, ions%pos, ions%atom, grp)

      if (outp%what(OPTION__OUTPUT__POTENTIAL_GRADIENT)) then
        SAFE_ALLOCATE(vh(1:der%mesh%np_part))
        SAFE_ALLOCATE(gradvh(1:der%mesh%np, 1:space%dim))
        vh(1:der%mesh%np) = hm%ks_pot%vhartree(1:der%mesh%np)
        call dderivatives_grad(der, vh, gradvh)
        call io_function_output_vector(outp%how(OPTION__OUTPUT__POTENTIAL_GRADIENT), dir, 'grad_vh', namespace, &
          space, der%mesh, gradvh(:, :), units_out%force, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        SAFE_DEALLOCATE_A(vh)

        SAFE_ALLOCATE(v0(1:der%mesh%np_part, 1))
        v0(1:der%mesh%np, 1) = hm%ep%vpsl(1:der%mesh%np)
        call dderivatives_grad(der, v0(1:der%mesh%np_part, 1), gradvh)
        call io_function_output_vector(outp%how(OPTION__OUTPUT__POTENTIAL_GRADIENT), dir, 'grad_v0', namespace, &
          space, der%mesh, gradvh(:, :), units_out%force, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        SAFE_DEALLOCATE_A(v0)
        SAFE_DEALLOCATE_A(gradvh)

      end if

      SAFE_ALLOCATE(potential(1:der%mesh%np_part))
      do is = 1, hm%d%nspin
        if (outp%what(OPTION__OUTPUT__POTENTIAL_GRADIENT)) then
          fname = get_filename_with_spin('grad_vxc', hm%d%nspin, is)
          SAFE_ALLOCATE(gradvh(1:der%mesh%np, 1:space%dim))
          potential(1:der%mesh%np) = hm%ks_pot%vxc(1:der%mesh%np, is)
          call dderivatives_grad(der, potential, gradvh)
          call io_function_output_vector(outp%how(OPTION__OUTPUT__POTENTIAL_GRADIENT), dir, fname, namespace, &
            space, der%mesh, gradvh(:, :), units_out%force, err, pos=ions%pos, atoms=ions%atom, grp = grp)
          SAFE_DEALLOCATE_A(gradvh)
        end if

        ! finally the full KS potential (without non-local PP contributions)
        potential(1:der%mesh%np) = hm%ep%vpsl + hm%ks_pot%vhxc(:, is)
        fname = get_filename_with_spin('vks', hm%d%nspin, is)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, fname, namespace, space, &
          der%mesh, potential, units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
      end do
      SAFE_DEALLOCATE_A(potential)
    end if

    !PCM potentials
    if (hm%theory_level == KOHN_SHAM_DFT .and. hm%pcm%run_pcm) then
      if (hm%pcm%solute .and. hm%pcm%localf) then
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm', namespace, space, &
          der%mesh, hm%pcm%v_e_rs + hm%pcm%v_n_rs + hm%pcm%v_ext_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_sol', namespace, space, &
          der%mesh, hm%pcm%v_e_rs + hm%pcm%v_n_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_e', namespace, space, &
          der%mesh, hm%pcm%v_e_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_n', namespace, space, &
          der%mesh, hm%pcm%v_n_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_ext', namespace, space, &
          der%mesh, hm%pcm%v_ext_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
      else if (hm%pcm%solute .and. .not. hm%pcm%localf) then
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_sol', namespace, space, &
          der%mesh, hm%pcm%v_e_rs + hm%pcm%v_n_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_e', namespace, space, &
          der%mesh, hm%pcm%v_e_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_n', namespace, space, &
          der%mesh, hm%pcm%v_n_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
      else if (.not. hm%pcm%solute .and. hm%pcm%localf) then
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'vpcm_ext', namespace, space, &
          der%mesh, hm%pcm%v_ext_rs , &
          units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)
      end if
    end if

    if (hm%self_induced_magnetic) then
      ! unit of magnetic field is same as of electric field, and same as force (since e = 1)
      select case (space%dim)
      case (3)
        do idir = 1, space%dim
          call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'Bind_'//index2axis(idir), namespace, &
            space, der%mesh, hm%b_ind(:, idir), &
            units_out%force, err, pos=ions%pos, atoms=ions%atom, grp = grp)
        end do
      case (2)
        call dio_function_output(outp%how(OPTION__OUTPUT__POTENTIAL), dir, 'Bind_z', namespace, &
          space, der%mesh, hm%b_ind(:, 1), units_out%force, err, pos=ions%pos, atoms=ions%atom, grp = grp)
      end select
    end if
  end if


  if (outp%what_now(OPTION__OUTPUT__XC_DENSITY, iter) .and. hm%theory_level /= INDEPENDENT_PARTICLES) then
    SAFE_ALLOCATE(v0(1:der%mesh%np_part, 1))
    SAFE_ALLOCATE(nxc(1:der%mesh%np))

    do is = 1, hm%d%nspin
      fname = get_filename_with_spin('nxc', hm%d%nspin, is)

      v0(1:der%mesh%np, 1) = hm%ks_pot%vxc(1:der%mesh%np, is)

      call dderivatives_lapl(der, v0(:, 1), nxc)

      call dio_function_output(outp%how(OPTION__OUTPUT__XC_DENSITY), dir, fname, namespace, &
        space, der%mesh, nxc, units_out%energy, err, pos=ions%pos, atoms=ions%atom, grp = grp)

    end do

    SAFE_DEALLOCATE_A(v0)
    SAFE_DEALLOCATE_A(nxc)
  end if

  if (outp%what_now(OPTION__OUTPUT__CURRENT, iter)) then

    if (states_are_complex(st)) then
      ASSERT(allocated(st%current))

      do is = 1, hm%d%nspin

        fname = get_filename_with_spin('current', st%d%nspin, is)

        call io_function_output_vector(outp%how(OPTION__OUTPUT__CURRENT), dir, fname, namespace, space, der%mesh, &
          st%current(:, :, is), (unit_one/units_out%time)*units_out%length**(1 - space%dim), err, &
          pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)

      end do
    else
      message(1) = 'No current density output for real states since it is identically zero.'
      call messages_warning(1, namespace=namespace)
    end if
  end if

  if (outp%what_now(OPTION__OUTPUT__CURRENT_DIA, iter)) then
    ASSERT(allocated(st%current_dia))
    do is = 1, hm%d%nspin
      fname = get_filename_with_spin('current_dia', st%d%nspin, is)

      call io_function_output_vector(outp%how(OPTION__OUTPUT__CURRENT_DIA), dir, fname, namespace, space, der%mesh, &
        st%current_dia(:, :, is), (unit_one/units_out%time)*units_out%length**(1 - space%dim), err, &
        pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)
    end do
  end if

  if (outp%what_now(OPTION__OUTPUT__CURRENT_KPT, iter)) then

    if (st%d%ispin == SPINORS) then
      call messages_not_implemented('current_kpt output option with spinors')
    end if

    if (states_are_complex(st)) then

      SAFE_ALLOCATE(current_kpt(st%d%kpt%start:st%d%kpt%end, 1:space%dim))
      do ik = st%d%kpt%start,st%d%kpt%end
        do idir = 1, space%dim
          current_kpt(ik, idir) = dmf_integrate(der%mesh, st%current_kpt(:, idir, ik), reduce = .false.)
        end do
      end do
      call der%mesh%allreduce(current_kpt, dim = (/st%d%kpt%end-st%d%kpt%start+1, space%dim/))

      write(fname, '(2a)') 'current_kpt'
      call io_function_output_vector_BZ(outp%how(OPTION__OUTPUT__CURRENT_KPT), dir, fname, namespace, space, &
        st%d%kpt, hm%kpoints, current_kpt(:, :), (unit_one/units_out%time)*units_out%length**(1 - space%dim), err, &
        grp = st%st_kpt_mpi_grp)
      SAFE_DEALLOCATE_A(current_kpt)
    else
      message(1) = 'No current density output for real states since it is identically zero.'
      call messages_warning(1, namespace=namespace)
    end if
  end if

  if (outp%what_now(OPTION__OUTPUT__HEAT_CURRENT, iter)) then

    if (states_are_complex(st)) then

      SAFE_ALLOCATE(heat_current(1:der%mesh%np_part, 1:space%dim, 1:st%d%nspin))

      call current_heat_calculate(space, der, hm, st, heat_current)

      do is = 1, hm%d%nspin
        fname = get_filename_with_spin('heat_current', st%d%nspin, is)

        call io_function_output_vector(outp%how(OPTION__OUTPUT__HEAT_CURRENT), dir, fname, namespace, space, der%mesh, &
          st%current(:, :, is), (unit_one/units_out%time)*units_out%length**(1 - space%dim), err, &
          pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)

        SAFE_DEALLOCATE_A(heat_current)
      end do
    else
      message(1) = 'No current density output for real states since it is identically zero.'
      call messages_warning(1, namespace=namespace)
    end if
  end if

  if (outp%what_now(OPTION__OUTPUT__DENSITY_KPT, iter)) then
    SAFE_ALLOCATE(density_kpt(1:st%nik, 1:st%d%nspin))
    density_kpt(1:st%nik, 1:st%d%nspin) = M_ZERO

    SAFE_ALLOCATE(density_tmp(1:gr%np, st%d%nspin))

    !Compute the k-resolved density and integrate it over the mesh
    do ik = st%d%kpt%start,st%d%kpt%end
      call density_calc_init(dens_calc, st, gr, density_tmp)
      do ib = st%group%block_start, st%group%block_end
        call density_calc_accumulate(dens_calc, st%group%psib(ib, ik))
      end do
      call density_calc_end(dens_calc, allreduce=.false., symmetrize=.false.)

      do is = 1, st%d%nspin
        density_kpt(ik, is) = density_kpt(ik, is) + dmf_integrate(der%mesh, density_tmp(:,is), reduce = .false.)
      end do
    end do

    call der%mesh%allreduce(density_kpt)

    if (st%parallel_in_states .or. st%d%kpt%parallel) then
      call comm_allreduce(st%dom_st_kpt_mpi_grp, density_kpt)
    end if

    if(mpi_grp_is_root(mpi_world)) then
      do is = 1, st%d%nspin
        fname = get_filename_with_spin('density_kpt', st%d%nspin, is)
        call io_function_output_global_BZ(outp%how(OPTION__OUTPUT__DENSITY_KPT), dir, fname, namespace, &
          hm%kpoints, density_kpt(:, is), unit_one, err)
      end do
    end if
    SAFE_DEALLOCATE_A(density_tmp)
    SAFE_DEALLOCATE_A(density_kpt)
  end if

  if(outp%what_now(OPTION__OUTPUT__EIGENVAL_KPT, iter)) then
    if(mpi_grp_is_root(mpi_world)) then
      do ist = 1, st%nst
        write(fname, '(a,i1)') 'eigenval_ist', ist
        call io_function_output_global_BZ(outp%how(OPTION__OUTPUT__EIGENVAL_KPT), dir, fname, namespace, &
          hm%kpoints, st%eigenval(ist, :), unit_one, err)
      end do
    end if
  end if

  POP_SUB(output_hamiltonian)
end subroutine output_hamiltonian


 ! ---------------------------------------------------------
subroutine output_scalar_pot(outp, namespace, space, dir, mesh, ions, ext_partners, time)
  type(output_t),           intent(in)    :: outp
  type(namespace_t),        intent(in)    :: namespace
  class(space_t),           intent(in)    :: space
  character(len=*),         intent(in)    :: dir
  class(mesh_t),            intent(in)    :: mesh
  type(ions_t),             intent(in)    :: ions
  type(partner_list_t),     intent(in)    :: ext_partners
  real(real64), optional,          intent(in)    :: time

  integer :: is, err
  character(len=80) :: fname
  real(real64), allocatable :: scalar_pot(:)
  type(lasers_t), pointer :: lasers

  PUSH_SUB(output_scalar_pot)

  if (outp%what(OPTION__OUTPUT__EXTERNAL_TD_POTENTIAL)) then
    SAFE_ALLOCATE(scalar_pot(1:mesh%np))
    lasers => list_get_lasers(ext_partners)
    if(associated(lasers)) then
      do is = 1, lasers%no_lasers
        write(fname, '(a,i1)') 'scalar_pot-', is
        scalar_pot = M_ZERO
        call laser_potential(lasers%lasers(is), mesh, scalar_pot, time=time)
        call dio_function_output(outp%how(OPTION__OUTPUT__EXTERNAL_TD_POTENTIAL), dir, fname, namespace, &
          space, mesh, scalar_pot, units_out%energy, err, pos=ions%pos, atoms=ions%atom)
      end do
    end if
    SAFE_DEALLOCATE_A(scalar_pot)
  end if

  POP_SUB(output_scalar_pot)
end subroutine output_scalar_pot


! ---------------------------------------------------------
subroutine output_xc_torque(outp, namespace, dir, mesh, hm, st, ions, space)
  type(output_t),           intent(in) :: outp
  type(namespace_t),        intent(in) :: namespace
  character(len=*),         intent(in) :: dir
  class(mesh_t),            intent(in) :: mesh
  type(hamiltonian_elec_t), intent(in) :: hm
  type(states_elec_t),      intent(in) :: st
  type(ions_t),             intent(in) :: ions
  class(space_t),           intent(in) :: space


  real(real64), allocatable :: torque(:,:)
  type(unit_t) :: fn_unit
  integer :: err

  PUSH_SUB(output_xc_torque)

  if (outp%what(OPTION__OUTPUT__XC_TORQUE)) then
    SAFE_ALLOCATE(torque(1:mesh%np, 1:3))

    call calc_xc_torque(mesh, hm%ks_pot%vxc, st, torque)

    fn_unit = units_out%length**(1 - 2*space%dim)
    call io_function_output_vector(outp%how(OPTION__OUTPUT__XC_TORQUE), dir, 'xc_torque', namespace, space, mesh, &
      torque, fn_unit, err, pos=ions%pos, atoms=ions%atom, grp = st%dom_st_kpt_mpi_grp)

    SAFE_DEALLOCATE_A(torque)
  end if

  POP_SUB(output_xc_torque)
end subroutine output_xc_torque


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
