!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module states_elec_calc_oct_m
  use accel_oct_m
  use accel_blas_oct_m
  use batch_oct_m
  use batch_ops_oct_m
  use blas_oct_m
  use blacs_oct_m
  use iso_c_binding
  use comm_oct_m
  use debug_oct_m
  use derivatives_oct_m
  use electron_space_oct_m
  use fourier_space_oct_m
  use global_oct_m
  use grid_oct_m
  use hardware_oct_m
  use kpoints_oct_m
  use lalg_adv_oct_m
  use lalg_basic_oct_m
  use math_oct_m
  use messages_oct_m
  use mesh_oct_m
  use mesh_batch_oct_m
  use mesh_function_oct_m
  use mpi_oct_m
  use mpi_lib_oct_m
  use namespace_oct_m
  use par_vec_oct_m
  use pblas_oct_m
  use physics_op_oct_m
  use poisson_oct_m
  use profiling_oct_m
  use scalapack_oct_m
  use singularity_oct_m
  use space_oct_m
  use states_abst_oct_m
  use states_elec_oct_m
  use states_elec_dim_oct_m
  use states_elec_parallel_oct_m
  use types_oct_m
  use wfs_elec_oct_m

  implicit none

  private

  public ::                              &
    states_elec_calc_norms,              &
    states_elec_orthogonalize,           &
    states_elec_rotate,                  &
    dstates_elec_calc_orth_test,         &
    zstates_elec_calc_orth_test,         &
    dstates_elec_orthogonalization,      &
    zstates_elec_orthogonalization,      &
    dstates_elec_orthogonalize_single,   &
    zstates_elec_orthogonalize_single,   &
    dstates_elec_orthogonalize_single_batch,   &
    zstates_elec_orthogonalize_single_batch,   &
    dstates_elec_orthogonalization_full, &
    zstates_elec_orthogonalization_full, &
    dstates_elec_residue,                &
    zstates_elec_residue,                &
    dstates_elec_matrix,                 &
    zstates_elec_matrix,                 &
    dstates_elec_calc_overlap,           &
    zstates_elec_calc_overlap,           &
    dstates_elec_calc_projections,       &
    zstates_elec_calc_projections,       &
    dstates_elec_rrqr_decomposition,     &
    zstates_elec_rrqr_decomposition

  interface states_elec_rotate
    module procedure dstates_elec_rotate, zstates_elec_rotate
  end interface states_elec_rotate

contains

  ! ---------------------------------------------------------
  !> @copydoc states_elec_calc_oct_m::dstates_elec_orthogonalization_full
  !
  subroutine states_elec_orthogonalize(st, namespace, mesh)
    type(states_elec_t),  intent(inout) :: st
    type(namespace_t),    intent(in)    :: namespace
    class(mesh_t),        intent(in)    :: mesh

    integer :: ik

    PUSH_SUB(states_elec_orthogonalize)

    do ik = st%d%kpt%start, st%d%kpt%end
      if (states_are_real(st)) then
        call dstates_elec_orthogonalization_full(st, namespace, mesh, ik)
      else
        call zstates_elec_orthogonalization_full(st, namespace, mesh, ik)
      end if
    end do

    POP_SUB(states_elec_orthogonalize)
  end subroutine states_elec_orthogonalize

  ! ---------------------------------------------------------
  !> @brief Compute the norms of the Kohn-Sham orbitals.
  !
  subroutine states_elec_calc_norms(grid, kpoints, st, norm_ks)
    type(grid_t),             intent(in)    :: grid          !< grid
    type(kpoints_t),          intent(in)    :: kpoints       !< kpoints
    type(states_elec_t),      intent(in)    :: st            !< KS states
    real(real64), contiguous,        intent(out)   :: norm_ks(:, :) !< Norm of KS wavefunctions/orbitals

    integer :: ik_ispin, iblock, minst, maxst

    PUSH_SUB(states_elec_calc_norms)

    ! If state or k is parallel, norm_ks is only partially touched, then gets summed over processes,
    ! hence initialise to zero
    norm_ks = M_ZERO

    do ik_ispin = st%d%kpt%start, st%d%kpt%end
      do iblock = st%group%block_start, st%group%block_end
        minst = states_elec_block_min(st, iblock)
        maxst = states_elec_block_max(st, iblock)
        ! Compute norm, defering grid communication until outside the loops
        call mesh_batch_nrm2(grid, st%group%psib(iblock, ik_ispin), norm_ks(minst:maxst, ik_ispin), reduce=.false.)
      enddo
    enddo

    ! Reduction over domains
    if (grid%parallel_in_domains) then
      norm_ks = norm_ks**2
      call grid%allreduce(norm_ks)
      norm_ks = sqrt(norm_ks)
    end if

    ! Reduction over states and k-points
    call comm_allreduce(st%st_kpt_mpi_grp, norm_ks)

    POP_SUB(states_elec_calc_norms)

  end subroutine states_elec_calc_norms


#include "undef.F90"
#include "real.F90"
#include "states_elec_calc_inc.F90"

#include "undef.F90"
#include "complex.F90"
#include "states_elec_calc_inc.F90"
#include "undef.F90"

end module states_elec_calc_oct_m


!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
