!!! Copyright (C) 2008-2010 David Strubbe
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

#include "global.h"

module kdotp_calc_oct_m
  use comm_oct_m
  use debug_oct_m
  use global_oct_m
  use grid_oct_m
  use hamiltonian_elec_oct_m
  use lalg_basic_oct_m
  use linear_response_oct_m
  use mesh_oct_m
  use mesh_function_oct_m
  use mesh_batch_oct_m
  use messages_oct_m
  use mpi_oct_m
  use namespace_oct_m
  use perturbation_oct_m
  use profiling_oct_m
  use space_oct_m
  use states_abst_oct_m
  use states_elec_oct_m
  use states_elec_calc_oct_m
  use utils_oct_m
  use wfs_elec_oct_m

  implicit none

  private
  public ::                        &
    dcalc_eff_mass_inv,            &
    zcalc_eff_mass_inv,            &
    calc_band_velocity,            &
    dkdotp_add_occ,                &
    zkdotp_add_occ,                &
    kdotp_wfs_tag

contains

! ---------------------------------------------------------
  character(len=100) function kdotp_wfs_tag(dir, dir2) result(str)
    integer,           intent(in) :: dir
    integer, optional, intent(in) :: dir2

    PUSH_SUB(kdotp_wfs_tag)

    str = "wfs_" // index2axis(dir)
    if (present(dir2)) str = trim(str) // "_" // index2axis(dir2)

    POP_SUB(kdotp_wfs_tag)

  end function kdotp_wfs_tag

  ! ---------------------------------------------------------
  !> Computes the k-point and band-resolved velocity.
  !!
  !! This is done using the following formula
  !! \f[
  !! v = (dE_nk/dk)/hbar = -Im < u_nk | -i grad | u_nk >
  !! \f]
  !!
  !! This is identically zero for real wavefunctions.
  subroutine calc_band_velocity(namespace, space, gr, st, hm, pert, velocity)
    type(namespace_t),        intent(in)    :: namespace
    class(space_t),           intent(in)    :: space
    type(grid_t),             intent(in)    :: gr
    type(states_elec_t),      intent(in)    :: st
    type(hamiltonian_elec_t), intent(inout) :: hm
    class(perturbation_t),    intent(inout) :: pert
    real(real64),             intent(out)   :: velocity(:,:,:)

    integer :: ik, ist, idir, ib, ind
    type(wfs_elec_t) :: pert_psib
    complex(real64) :: dot(st%block_size)

    PUSH_SUB(kdotp_calc_band_velocity)

    velocity(:, :, :) = M_ZERO
    if(states_are_real(st)) then
      POP_SUB(kdotp_calc_band_velocity)
      return
    end if

    call profiling_in("CALC_BAND_VELOCITY")

    do ik = st%d%kpt%start, st%d%kpt%end
      do ib = st%group%block_start, st%group%block_end

        call st%group%psib(ib, ik)%copy_to(pert_psib)

        do idir = 1, space%periodic_dim
          call pert%setup_dir(idir)
          call pert%apply_batch(namespace, space, gr, hm, st%group%psib(ib, ik), pert_psib)

          call zmesh_batch_dotp_vector(gr, st%group%psib(ib, ik), pert_psib,  dot)

          do ind = 1, pert_psib%nst
            ist = st%group%psib(ib, ik)%ist(ind)
            velocity(idir, ist, ik) = -aimag(dot(ind))
          end do
        end do
        call pert_psib%end()
      end do
    end do

    if (st%parallel_in_states .or. st%d%kpt%parallel) then
      call comm_allreduce(st%st_kpt_mpi_grp, velocity)
    end if

    call profiling_out("CALC_BAND_VELOCITY")

    POP_SUB(kdotp_calc_band_velocity)
  end subroutine calc_band_velocity

#include "undef.F90"
#include "real.F90"
#include "kdotp_calc_inc.F90"

#include "undef.F90"
#include "complex.F90"
#include "kdotp_calc_inc.F90"

end module kdotp_calc_oct_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
