!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!

! ---------------------------------------------------------
!> This routine calculates the SIC exchange functional.
subroutine X(oep_sic) (xcs, gr, psolver, namespace, space, rcell_volume, st, kpoints, is, oep, ex, ec)
  type(xc_t),          intent(inout) :: xcs
  type(grid_t),        intent(in)    :: gr
  type(poisson_t),     intent(in)    :: psolver
  type(namespace_t),   intent(in)    :: namespace
  class(space_t),      intent(in)    :: space
  real(real64),        intent(in)    :: rcell_volume
  type(states_elec_t), intent(inout) :: st
  type(kpoints_t),     intent(in)    :: kpoints
  integer,             intent(in)    :: is
  type(xc_oep_t),      intent(inout) :: oep
  real(real64),        intent(inout) :: ex, ec

  integer  :: ist
  real(real64) :: ex2, ec2, ex_, ec_
  real(real64), allocatable :: vxc(:, :), rho(:,:)
  R_TYPE, allocatable :: psi(:, :)

  call profiling_in(TOSTRING(X(XC_SIC)))
  PUSH_SUB(X(oep_sic))

  if (xc_is_not_size_consistent(xcs, namespace)) then
    call messages_not_implemented('PZ-SIC with size inconsistent functionals',  namespace=namespace)
  end if


  ASSERT(st%d%dim == 1)
  ASSERT(.not. st%nik > st%d%spin_channels)

  SAFE_ALLOCATE(psi(1:gr%np, 1:st%d%dim))
  SAFE_ALLOCATE(rho(1:gr%np, 1:2))
  SAFE_ALLOCATE(vxc(1:gr%np, 1:2))
  rho(1:gr%np, 2) = M_ZERO

  ! loop over states
  ex_ = M_ZERO
  ec_ = M_ZERO
  do ist = st%st_start, st%st_end
    if (st%occ(ist, is) <= M_MIN_OCC) cycle ! we only need the occupied states

    call states_elec_get_state(st, gr, ist, is, psi)

    ! get orbital density
    rho(1:gr%np, 1) = R_REAL(psi(1:gr%np, 1)*R_CONJ(psi(1:gr%np, 1)))

    ! initialize before calling get_vxc
    vxc = M_ZERO
    ex2  = M_ZERO
    ec2  = M_ZERO

    ! calculate LDA/GGA contribution to the SIC (does not work for LB94)
    call xc_get_vxc(gr, xcs, st, kpoints, psolver, namespace, space, rho, SPIN_POLARIZED, rcell_volume, vxc, ex=ex2, ec=ec2)

    ex_ = ex_ - oep%sfact*ex2*oep%socc*st%occ(ist, is)
    ec_ = ec_ - oep%sfact*ec2*oep%socc*st%occ(ist, is)

    oep%X(lxc)(1:gr%np, 1, ist, is) = oep%X(lxc)(1:gr%np, 1, ist, is) - vxc(1:gr%np, 1)*R_CONJ(psi(1:gr%np, 1))*oep%socc*st%occ(ist, is)

    ! calculate the Hartree contribution using Poisson equation
    vxc(1:gr%np, 1) = M_ZERO
    call dpoisson_solve(psolver, namespace, vxc(:, 1), rho(:, 1), all_nodes=.false.)

    ! The exchange energy.
    ex_ = ex_ - M_HALF*oep%sfact*oep%socc*st%occ(ist, is)*dmf_dotp(gr, vxc(1:gr%np, 1), rho(1:gr%np, 1))

    oep%X(lxc)(1:gr%np, 1, ist, is) = oep%X(lxc)(1:gr%np, 1, ist, is) - &
      vxc(1:gr%np, 1)*R_CONJ(psi(1:gr%np, 1))*oep%socc*st%occ(ist, is)
  end do

  if (st%parallel_in_states) then
    call comm_allreduce(st%mpi_grp, ec_)
    call comm_allreduce(st%mpi_grp, ex_)
  end if

  ec = ec + ec_
  ex = ex + ex_

  SAFE_DEALLOCATE_A(rho)
  SAFE_DEALLOCATE_A(vxc)
  SAFE_DEALLOCATE_A(psi)

  POP_SUB(X(oep_sic))
  call profiling_out(TOSTRING(X(XC_SIC)))
end subroutine X(oep_sic)

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
