/**
 * SPDX-FileCopyrightText: (C) 2025 Francesco Pretto <ceztko@gmail.com>
 * SPDX-License-Identifier: LGPL-2.0-or-later
 * SPDX-License-Identifier: MPL-2.0
 */

#include <PdfTest.h>

using namespace std;
using namespace PoDoFo;

constexpr string_view diamonds = u8"\u2666";
constexpr string_view    clubs = u8"\u2663";
constexpr string_view   spades = u8"\u2660";
constexpr string_view   hearts = u8"\u2665";

// ISO 32000-2:2020 8.7.3.2 "Coloured tiling patterns", example 2
TEST_CASE("TestColouredTilingPattern")
{
    PdfMemDocument doc;
    auto& page = doc.GetPages().CreatePage();

    auto colouredPatternDef = std::make_shared<PdfColouredTilingPatternDefinition>(PdfTilingSpacingType::NoDistortion,
        Rect(0, 0, 100, 100), 100, 100, Matrix(0.4, 0, 0, 0.4, 0, 0));
    auto colouredPattern = doc.CreateTilingPattern(colouredPatternDef);

    auto& zapfDingbats = doc.GetFonts().GetStandard14Font(PdfStandard14FontType::ZapfDingbats);

    {
        PdfPainter painter;
        painter.SetCanvas(*colouredPattern);
        auto& operators = static_cast<PdfContentStreamOperators&>(painter);
        painter.TextObject.Begin();
        painter.TextState.SetFont(zapfDingbats, 1);
        painter.TextState.SetMatrix(Matrix(64, 0, 0, 64, 7.1771, 2.4414));
        painter.TextState.SetCharSpacing(0);
        painter.TextState.SetWordSpacing(0);
        painter.GraphicsState.SetNonStrokingColor(PdfColor(1, 0, 0));
        painter.TextObject.AddText(spades);
        //painter.TextObject.MoveTo(0.7478, -0.007);
        operators.TD_Operator(0.7478, -0.007);
        painter.GraphicsState.SetNonStrokingColor(PdfColor(0, 1, 0));
        painter.TextObject.AddText(hearts);
        //painter.TextObject.MoveTo(-0.7323, 0.7813);
        operators.TD_Operator(-0.7323, 0.7813);
        painter.GraphicsState.SetNonStrokingColor(PdfColor(0, 0, 1));
        painter.TextObject.AddText(diamonds);
        //painter.TextObject.MoveTo(0.6913, 0.007);
        operators.TD_Operator(0.6913, 0.007);
        painter.GraphicsState.SetNonStrokingColor(PdfColor(0, 0, 0));
        painter.TextObject.AddText(clubs);
        painter.TextObject.End();
    }

    {
        PdfPainter painter;
        painter.SetCanvas(page);
        auto& operators = static_cast<PdfContentStreamOperators&>(painter);

        painter.GraphicsState.SetStrokingColor(PdfColor(0));
        painter.GraphicsState.SetNonStrokingColor(PdfColor(1, 1, 0));

        // NOTE: Coordinates/coefficients were extracted from PdfReference 1.7 page 1143.
        // Coordinates/coefficients in ISO 32000-2:2020 are wrong
        painter.DrawRectangle(Rect(272.22, 442.087, 131.255, 112.504), PdfPathDrawMode::Fill);

        painter.GraphicsState.SetNonStrokingPattern(*colouredPattern);

        operators.m_Operator(328.503, 460.822);
        operators.c_Operator(328.503, 481.523, 311.702, 498.324, 291.001, 498.324);
        operators.c_Operator(270.3, 498.324, 253.5, 481.523, 253.5, 460.822);
        operators.c_Operator(253.5, 440.121, 270.3, 423.321, 291.001, 423.321);
        operators.c_Operator(311.702, 423.321, 328.503, 440.121, 328.503, 460.822);
        operators.B_Operator();

        operators.m_Operator(422.196, 460.822);
        operators.c_Operator(422.196, 481.523, 405.396, 498.324, 384.695, 498.324);
        operators.c_Operator(363.994, 498.324, 347.193, 481.523, 347.193, 460.822);
        operators.c_Operator(347.193, 440.121, 363.994, 423.321, 384.695, 423.321);
        operators.c_Operator(405.396, 423.321, 422.196, 440.121, 422.196, 460.822);
        operators.B_Operator();

        operators.m_Operator(319.097, 574.482);
        operators.c_Operator(301.17, 564.132, 295.021, 541.182, 305.371, 523.255);
        operators.c_Operator(315.721, 505.328, 338.672, 499.178, 356.599, 509.528);
        operators.c_Operator(374.526, 519.879, 380.675, 542.829, 370.325, 560.756);
        operators.c_Operator(359.975, 578.683, 337.024, 584.833, 319.097, 574.482);
        operators.B_Operator();

        operators.m_Operator(290.971, 460.837);
        operators.l_Operator(384.725, 460.837);
        operators.l_Operator(337.848, 542.03);
        operators.b_Operator();
    }

    doc.Save(TestUtils::GetTestOutputFilePath("TestColouredPattern.pdf"), PdfSaveOptions::NoMetadataUpdate);
    TestUtils::AreFilesEqual(TestUtils::GetTestOutputFilePath("TestColouredPattern.pdf"), TestUtils::GetTestInputFilePath("TestColouredPatternRef.pdf"));
}

// ISO 32000-2:2020 8.7.3.3 "Uncoloured tiling patterns", example 3
TEST_CASE("TestUncolouredTilingPattern")
{
    PdfMemDocument doc;
    auto& page = doc.GetPages().CreatePage();

    auto colouredPatternDef = std::make_shared<PdfUncolouredTilingPatternDefinition>(PdfTilingSpacingType::NoDistortion,
        Rect(0, 0, 100, 100), 100, 100, Matrix(0.4, 0, 0, 0.4, 0, 0));
    auto uncolouredPattern = doc.CreateTilingPattern(colouredPatternDef);

    auto& zapfDingbats = doc.GetFonts().GetStandard14Font(PdfStandard14FontType::ZapfDingbats);

    {
        PdfPainter painter;
        painter.SetCanvas(*uncolouredPattern);
        painter.TextObject.Begin();
        painter.TextState.SetFont(zapfDingbats, 1);
        painter.TextState.SetMatrix(Matrix(64, 0, 0, 64, 7.1771, 2.4414));
        painter.TextState.SetCharSpacing(0);
        painter.TextState.SetWordSpacing(0);
        painter.TextObject.AddText(spades);
        painter.TextObject.MoveTo(0.7478, -0.007);
        painter.TextObject.AddText(hearts);
        painter.TextObject.MoveTo(-0.7323, 0.7813);
        painter.TextObject.AddText(diamonds);
        painter.TextObject.MoveTo(0.6913, 0.007);
        painter.TextObject.AddText(clubs);
        painter.TextObject.End();
    }

    {
        PdfPainter painter;
        painter.SetCanvas(page);
        auto& operators = static_cast<PdfContentStreamOperators&>(painter);

        painter.GraphicsState.SetStrokingColor(PdfColor(0));
        painter.GraphicsState.SetNonStrokingColor(PdfColor(1, 1, 0));

        // NOTE: Coordinates/coefficients were extracted from PdfReference 1.7 page 1143.
        // Coordinates/coefficients in ISO 32000-2:2020 are wrong
        painter.DrawRectangle(Rect(272.22, 172.075, 131.255, 112.504), PdfPathDrawMode::Fill);

        auto patternColorspace = doc.CreateColorSpace(std::make_unique<PdfColorSpaceFilterPattern>(PdfColorSpaceType::DeviceRGB));

        painter.GraphicsState.SetNonStrokingColorSpace(*patternColorspace);

        painter.GraphicsState.SetNonStrokingUncolouredTilingPattern(*uncolouredPattern, PdfColorRaw{ 0.77, 0.20, 0.00 });
        operators.m_Operator(328.503, 190.811);
        operators.c_Operator(328.503, 211.512, 311.702, 228.312, 291.001, 228.312);
        operators.c_Operator(270.3, 228.312, 253.5, 211.512, 253.5, 190.811);
        operators.c_Operator(253.5, 170.11, 270.3, 153.309, 291.001, 153.309);
        operators.c_Operator(311.702, 153.309, 328.503, 170.11, 328.503, 190.811);
        operators.B_Operator();

        painter.GraphicsState.SetNonStrokingUncolouredTilingPattern(*uncolouredPattern, PdfColorRaw{ 0.2, 0.8, 0.4 });
        operators.m_Operator(422.196, 190.811);
        operators.c_Operator(422.196, 211.512, 405.396, 228.312, 384.695, 228.312);
        operators.c_Operator(363.994, 228.312, 347.193, 211.512, 347.193, 190.811);
        operators.c_Operator(347.193, 170.11, 363.994, 153.309, 384.695, 153.309);
        operators.c_Operator(405.396, 153.309, 422.196, 170.11, 422.196, 190.811);
        operators.B_Operator();

        painter.GraphicsState.SetNonStrokingUncolouredTilingPattern(*uncolouredPattern, PdfColorRaw{ 0.3, 0.7, 1.0 });
        operators.m_Operator(319.097, 304.471);
        operators.c_Operator(301.17, 294.121, 295.021, 271.171, 305.371, 253.243);
        operators.c_Operator(315.721, 235.316, 338.672, 229.167, 356.599, 239.517);
        operators.c_Operator(374.526, 249.867, 380.675, 272.818, 370.325, 290.745);
        operators.c_Operator(359.975, 308.672, 337.024, 314.822, 319.097, 304.471);
        operators.B_Operator();

        painter.GraphicsState.SetNonStrokingUncolouredTilingPattern(*uncolouredPattern, PdfColorRaw{ 0.5, 0.2, 1.0 });
        operators.m_Operator(290.971, 190.826);
        operators.l_Operator(384.725, 190.826);
        operators.l_Operator(337.848, 272.019);
        operators.b_Operator();
    }

    doc.Save(TestUtils::GetTestOutputFilePath("TestUncolouredPattern.pdf"), PdfSaveOptions::NoMetadataUpdate);
    TestUtils::AreFilesEqual(TestUtils::GetTestOutputFilePath("TestUncolouredPattern.pdf"), TestUtils::GetTestInputFilePath("TestUncolouredPatternRef.pdf"));
}

// ISO 32000-2:2020 8.7.4.5.4 "Type 3 (radial) shadings", example 1
TEST_CASE("TestType3RadialShading")
{
    PdfMemDocument doc;
    auto& page = doc.GetPages().CreatePage();

    auto expFoo1 = doc.CreateFunction(std::make_shared<PdfExponentialFunctionDefinition>(1.048, vector<double>{ 0, 1 },
        vector<double>{ 0.929, 0.357, 1, 0.298 }, vector<double>{ 0.631, 0.278, 1, 0.027 }));

    auto expFoo2 = doc.CreateFunction(std::make_shared<PdfExponentialFunctionDefinition>(1.374, vector<double>{ 0, 1 },
        vector<double>{ 0.929, 0.357, 1, 0.298 }, vector<double>{ 0.941, 0.400, 1, 0.102 }));

    auto stitchingFoo = doc.CreateFunction(PdfFunctionDefinitionPtr(new PdfStitchingFunctionDefinition(
        { *expFoo1, *expFoo2 }, vector<double>{ 0.708 }, vector<double>{ 1, 0, 0, 1 }, vector<double>{ 0, 1 })));

    auto radialShadingDef = std::make_shared<PdfRadialShadingDefinition>(PdfColorSpaceType::DeviceCMYK,
        *stitchingFoo, array<double, 6>{ 0, 0, 0.096, 0, 0, 1 },
        array<bool, 2>{ true, true });
    auto radialShading = doc.CreateShadingDictionary(radialShadingDef);

    {
        PdfPainter painter;
        painter.SetCanvas(page);
        auto& operators = static_cast<PdfContentStreamOperators&>(painter);

        // NOTE: The original and full operator sequence were extracted from
        // PdfReference 1.3 page 224, figure 4.19 "Radial shading"
        operators.rg_Operator(1, 1, 1);
        operators.re_Operator(65, 510, 400, -116);
        operators.f_Operator();
        painter.Save();
        operators.i_Operator(1.5);
        operators.m_Operator(242.291, 462.058);
        operators.c_Operator(248.836, 480.87, 276.84, 471.17, 287.068, 486.61);
        operators.c_Operator(284.679, 480.981, 278.568, 476.183, 274.216, 471.964);
        operators.c_Operator(270.646, 468.504, 266.368, 464.644, 261.764, 462.685);
        operators.c_Operator(255.153, 459.873, 244.757, 461.715, 239.554, 457.18);
        operators.l_Operator(242.291, 462.058);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.4592, -19.6178, -19.6179, -3.4592, 263.3156, 471.9249);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.K_Operator(1, 0.61, 0.6, 0.322);
        operators.w_Operator(0.15);
        operators.M_Operator(4);
        operators.i_Operator(1.5);
        operators.m_Operator(242.291, 462.058);
        operators.c_Operator(248.836, 480.87, 276.84, 471.17, 287.068, 486.61);
        operators.c_Operator(284.679, 480.981, 278.568, 476.183, 274.216, 471.964);
        operators.c_Operator(270.646, 468.504, 266.368, 464.644, 261.764, 462.685);
        operators.c_Operator(255.153, 459.873, 244.757, 461.715, 239.554, 457.18);
        operators.l_Operator(242.291, 462.058);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(114.487, 462.046);
        operators.c_Operator(129.283, 454.84, 142.087, 452.744, 146.875, 433.949);
        operators.c_Operator(150.545, 419.547, 146.444, 405.587, 135.049, 396.36);
        operators.c_Operator(138.571, 411.858, 131.352, 426.461, 123.087, 439.188);
        operators.c_Operator(118.853, 445.706, 111.221, 453.809, 116.417, 461.472);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.5813, -25.982, -25.982, -4.5813, 133.3193, 430.5479);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(114.487, 462.046);
        operators.c_Operator(129.283, 454.84, 142.087, 452.744, 146.875, 433.949);
        operators.c_Operator(150.545, 419.547, 146.444, 405.587, 135.049, 396.36);
        operators.c_Operator(138.571, 411.858, 131.352, 426.461, 123.087, 439.188);
        operators.c_Operator(118.853, 445.706, 111.221, 453.809, 116.417, 461.472);
        operators.S_Operator();
        painter.Save();
        operators.m_Operator(48.559, 452.168);
        operators.c_Operator(58.127, 462.778, 79.015, 432.983, 81.468, 425.393);
        operators.c_Operator(86.609, 409.475, 84.144, 391.612, 76.014, 378.052);
        operators.c_Operator(65.313, 390.959, 63.003, 405.083, 63.767, 420.887);
        operators.c_Operator(64.047, 426.689, 56.708, 464.698, 47.856, 453.825);
        operators.l_Operator(48.559, 452.168);
        operators.W_Operator();
        operators.n_Operator();
        operators.re_Operator(65, 510, 400, -116);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(11.8842, -28.0642, -28.0643, -11.8842, 69.9543, 419.6993);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        painter.Save();
        operators.re_Operator(65, 510, 400, -116);
        operators.W_Operator();
        operators.n_Operator();
        operators.m_Operator(48.559, 452.168);
        operators.c_Operator(58.127, 462.778, 79.015, 432.983, 81.468, 425.393);
        operators.c_Operator(86.609, 409.475, 84.144, 391.612, 76.014, 378.052);
        operators.c_Operator(65.313, 390.959, 63.003, 405.083, 63.767, 420.887);
        operators.c_Operator(64.047, 426.689, 56.708, 464.698, 47.856, 453.825);
        operators.l_Operator(48.559, 452.168);
        operators.s_Operator();
        painter.Restore();
        painter.Save();
        operators.m_Operator(-36.658, 444.853);
        operators.c_Operator(-18.898, 445.596, -1.832, 449.7, 16.176, 451.271);
        operators.c_Operator(32.96, 452.734, 50.047, 451.593, 67.022, 452.878);
        operators.c_Operator(82.196, 454.027, 99.741, 456.932, 114.072, 457.245);
        operators.l_Operator(117.492, 457.295);
        operators.c_Operator(121.861, 457.623, 139.789, 455.51, 142.399, 454.792);
        operators.c_Operator(156.228, 450.998, 169.361, 447.357, 183.845, 448.409);
        operators.c_Operator(194.921, 449.213, 205.925, 451.935, 217.075, 452.805);
        operators.c_Operator(230.972, 453.889, 244.276, 455.363, 258.321, 452.767);
        operators.c_Operator(273.985, 449.87, 287.593, 447.977, 303.521, 449.313);
        operators.c_Operator(317.495, 450.485, 331.54, 452.808, 345.628, 453.873);
        operators.c_Operator(360.663, 455.01, 375.426, 455.659, 390.441, 457.776);
        operators.c_Operator(404.899, 459.811, 420.534, 460.943, 435.164, 460.634);
        operators.l_Operator(436.13, 462.887);
        operators.c_Operator(421.514, 463.572, 405.855, 462.844, 391.349, 461.18);
        operators.c_Operator(376.285, 459.452, 361.509, 459.182, 346.451, 458.434);
        operators.c_Operator(332.341, 457.731, 318.239, 455.77, 304.24, 454.959);
        operators.c_Operator(288.283, 454.034, 274.728, 456.276, 259.145, 459.575);
        operators.c_Operator(245.171, 462.532, 231.833, 461.401, 217.913, 460.674);
        operators.c_Operator(206.744, 460.091, 195.673, 457.655, 184.581, 457.137);
        operators.c_Operator(170.074, 456.457, 157.039, 460.435, 143.311, 464.586);
        operators.c_Operator(140.73, 465.366, 118.707, 467.357, 114.534, 467.175);
        operators.c_Operator(99.406, 466.514, 82.307, 463.755, 67.136, 462.606);
        operators.c_Operator(50.158, 461.321, 33.071, 462.463, 16.29, 460.999);
        operators.c_Operator(-1.721, 459.428, -18.787, 455.324, -36.544, 454.581);
        operators.c_Operator(-55.642, 453.779, -74.988, 452.99, -94.017, 448.813);
        operators.c_Operator(-94.277, 448.756, -94.534, 448.691, -94.794, 448.632);
        operators.l_Operator(-93.146, 439.289);
        operators.c_Operator(-73.76, 443.425, -54.768, 444.094, -36.658, 444.853);
        operators.W_Operator();
        operators.n_Operator();
        operators.re_Operator(65, 510, 400, -116);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(46.2701, -262.4107, -262.4112, -46.2702, 60.4724, 443.5012);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        painter.Save();
        operators.re_Operator(65, 510, 400, -116);
        operators.W_Operator();
        operators.n_Operator();
        operators.m_Operator(-36.658, 444.853);
        operators.c_Operator(-18.898, 445.596, -1.832, 449.7, 16.176, 451.271);
        operators.c_Operator(32.96, 452.734, 50.047, 451.593, 67.022, 452.878);
        operators.c_Operator(82.196, 454.027, 99.741, 456.932, 114.072, 457.245);
        operators.l_Operator(117.492, 457.295);
        operators.c_Operator(121.861, 457.623, 139.789, 455.51, 142.399, 454.792);
        operators.c_Operator(156.228, 450.998, 169.361, 447.357, 183.845, 448.409);
        operators.c_Operator(194.921, 449.213, 205.925, 451.935, 217.075, 452.805);
        operators.c_Operator(230.972, 453.889, 244.276, 455.363, 258.321, 452.767);
        operators.c_Operator(273.985, 449.87, 287.593, 447.977, 303.521, 449.313);
        operators.c_Operator(317.495, 450.485, 331.54, 452.808, 345.628, 453.873);
        operators.c_Operator(360.663, 455.01, 375.426, 455.659, 390.441, 457.776);
        operators.c_Operator(404.899, 459.811, 420.534, 460.943, 435.164, 460.634);
        operators.l_Operator(436.13, 462.887);
        operators.c_Operator(421.514, 463.572, 405.855, 462.844, 391.349, 461.18);
        operators.c_Operator(376.285, 459.452, 361.509, 459.182, 346.451, 458.434);
        operators.c_Operator(332.341, 457.731, 318.239, 455.77, 304.24, 454.959);
        operators.c_Operator(288.283, 454.034, 274.728, 456.276, 259.145, 459.575);
        operators.c_Operator(245.171, 462.532, 231.833, 461.401, 217.913, 460.674);
        operators.c_Operator(206.744, 460.091, 195.673, 457.655, 184.581, 457.137);
        operators.c_Operator(170.074, 456.457, 157.039, 460.435, 143.311, 464.586);
        operators.c_Operator(140.73, 465.366, 118.707, 467.357, 114.534, 467.175);
        operators.c_Operator(99.406, 466.514, 82.307, 463.755, 67.136, 462.606);
        operators.c_Operator(50.158, 461.321, 33.071, 462.463, 16.29, 460.999);
        operators.c_Operator(-1.721, 459.428, -18.787, 455.324, -36.544, 454.581);
        operators.c_Operator(-55.642, 453.779, -74.988, 452.99, -94.017, 448.813);
        operators.c_Operator(-94.277, 448.756, -94.534, 448.691, -94.794, 448.632);
        operators.l_Operator(-93.146, 439.289);
        operators.c_Operator(-73.76, 443.425, -54.768, 444.094, -36.658, 444.853);
        operators.s_Operator();
        painter.Restore();
        painter.Save();
        operators.m_Operator(308.019, 451.267);
        operators.c_Operator(322.238, 458.48, 341.697, 448.933, 351.199, 437.62);
        operators.c_Operator(355.232, 432.82, 357.632, 426.883, 360.161, 421.365);
        operators.c_Operator(361.647, 418.123, 361.801, 416.978, 360.532, 413.485);
        operators.c_Operator(355.864, 420.908, 349.322, 425.338, 341.519, 429.5);
        operators.c_Operator(334.967, 432.994, 327.848, 436.345, 322.587, 441.754);
        operators.c_Operator(320.983, 443.404, 314.984, 456.158, 311.827, 452.339);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.1253, -23.3955, -23.3955, -4.1253, 335.1019, 432.8977);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(308.019, 451.267);
        operators.c_Operator(322.238, 458.48, 341.697, 448.933, 351.199, 437.62);
        operators.c_Operator(355.232, 432.82, 357.632, 426.883, 360.161, 421.365);
        operators.c_Operator(361.647, 418.123, 361.801, 416.978, 360.532, 413.485);
        operators.c_Operator(355.864, 420.908, 349.322, 425.338, 341.519, 429.5);
        operators.c_Operator(334.967, 432.994, 327.848, 436.345, 322.587, 441.754);
        operators.c_Operator(320.983, 443.404, 314.984, 456.158, 311.827, 452.339);
        operators.S_Operator();
        painter.Save();
        operators.m_Operator(287.524, 466.189);
        operators.c_Operator(297.274, 477.492, 323.37, 472.871, 327.313, 490.059);
        operators.c_Operator(322.576, 483.81, 321.402, 472.018, 312.382, 467.767);
        operators.c_Operator(309.739, 466.521, 306.701, 466.651, 303.875, 465.875);
        operators.c_Operator(301.799, 465.306, 299.113, 462.803, 298.101, 462.859);
        operators.l_Operator(287.524, 466.189);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.0188, -17.1204, -17.1204, -3.0188, 307.7633, 475.6076);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(287.524, 466.189);
        operators.c_Operator(297.274, 477.492, 323.37, 472.871, 327.313, 490.059);
        operators.c_Operator(322.576, 483.81, 321.402, 472.018, 312.382, 467.767);
        operators.c_Operator(309.739, 466.521, 306.701, 466.651, 303.875, 465.875);
        operators.c_Operator(301.799, 465.306, 299.113, 462.803, 298.101, 462.859);
        operators.l_Operator(287.524, 466.189);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(266.039, 458.836);
        operators.c_Operator(275.729, 463.825, 282.806, 467.843, 294.196, 466.999);
        operators.c_Operator(299.744, 466.587, 304.922, 465.763, 310.378, 464.756);
        operators.c_Operator(315.568, 463.797, 320.199, 464.79, 325.453, 465.716);
        operators.c_Operator(320.478, 464.839, 315.562, 458.602, 311.456, 455.744);
        operators.c_Operator(305.252, 451.425, 298.265, 450.693, 290.941, 450.458);
        operators.c_Operator(285.105, 450.272, 279.313, 450.296, 273.905, 452.641);
        operators.c_Operator(271.222, 453.803, 266.938, 459.085, 264.944, 458.244);
        operators.l_Operator(266.039, 458.836);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.9101, -22.1755, -22.1756, -3.9102, 295.6753, 459.4965);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(266.039, 458.836);
        operators.c_Operator(275.729, 463.825, 282.806, 467.843, 294.196, 466.999);
        operators.c_Operator(299.744, 466.587, 304.922, 465.763, 310.378, 464.756);
        operators.c_Operator(315.568, 463.797, 320.199, 464.79, 325.453, 465.716);
        operators.c_Operator(320.478, 464.839, 315.562, 458.602, 311.456, 455.744);
        operators.c_Operator(305.252, 451.425, 298.265, 450.693, 290.941, 450.458);
        operators.c_Operator(285.105, 450.272, 279.313, 450.296, 273.905, 452.641);
        operators.c_Operator(271.222, 453.803, 266.938, 459.085, 264.944, 458.244);
        operators.l_Operator(266.039, 458.836);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(275.782, 451.179);
        operators.c_Operator(280.099, 451.121, 284.65, 448.809, 289.161, 448.233);
        operators.c_Operator(295.899, 447.372, 300.564, 447.104, 306.067, 442.618);
        operators.c_Operator(310.996, 438.601, 313.881, 433.001, 318.518, 428.73);
        operators.c_Operator(321.951, 425.568, 326.119, 424.24, 329.657, 421.474);
        operators.c_Operator(323.486, 422.446, 318.185, 425.641, 312.549, 427.993);
        operators.c_Operator(306.326, 430.588, 299.855, 432.629, 293.574, 435.239);
        operators.c_Operator(288.484, 437.355, 281.665, 439.139, 277.897, 443.228);
        operators.c_Operator(275.374, 445.968, 275.53, 451.059, 274.346, 452.524);
        operators.l_Operator(275.782, 451.179);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.9648, -22.4857, -22.4857, -3.9648, 302.3117, 436.899);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(275.782, 451.179);
        operators.c_Operator(280.099, 451.121, 284.65, 448.809, 289.161, 448.233);
        operators.c_Operator(295.899, 447.372, 300.564, 447.104, 306.067, 442.618);
        operators.c_Operator(310.996, 438.601, 313.881, 433.001, 318.518, 428.73);
        operators.c_Operator(321.951, 425.568, 326.119, 424.24, 329.657, 421.474);
        operators.c_Operator(323.486, 422.446, 318.185, 425.641, 312.549, 427.993);
        operators.c_Operator(306.326, 430.588, 299.855, 432.629, 293.574, 435.239);
        operators.c_Operator(288.484, 437.355, 281.665, 439.139, 277.897, 443.228);
        operators.c_Operator(275.374, 445.968, 275.53, 451.059, 274.346, 452.524);
        operators.l_Operator(275.782, 451.179);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(239.05, 455.826);
        operators.c_Operator(245.996, 456.059, 253.168, 452.055, 257.563, 446.699);
        operators.c_Operator(260.907, 442.621, 261.919, 437.375, 265.447, 433.36);
        operators.c_Operator(268.302, 430.112, 271.776, 427.448, 275.11, 424.775);
        operators.c_Operator(271.511, 425.785, 269.04, 428.168, 266.005, 430.182);
        operators.c_Operator(262.589, 432.449, 258.856, 434.171, 255.175, 436.072);
        operators.c_Operator(249.419, 439.046, 243.476, 440.083, 237.425, 442.02);
        operators.c_Operator(232.527, 443.59, 230.634, 447.678, 226.773, 450.687);
        operators.c_Operator(224.032, 452.823, 221.049, 452.551, 218.584, 454.246);
        operators.l_Operator(239.05, 455.826);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(0.005, -23.1232, -23.1232, -0.005, 245.8655, 440.3891);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(239.05, 455.826);
        operators.c_Operator(245.996, 456.059, 253.168, 452.055, 257.563, 446.699);
        operators.c_Operator(260.907, 442.621, 261.919, 437.375, 265.447, 433.36);
        operators.c_Operator(268.302, 430.112, 271.776, 427.448, 275.11, 424.775);
        operators.c_Operator(271.511, 425.785, 269.04, 428.168, 266.005, 430.182);
        operators.c_Operator(262.589, 432.449, 258.856, 434.171, 255.175, 436.072);
        operators.c_Operator(249.419, 439.046, 243.476, 440.083, 237.425, 442.02);
        operators.c_Operator(232.527, 443.59, 230.634, 447.678, 226.773, 450.687);
        operators.c_Operator(224.032, 452.823, 221.049, 452.551, 218.584, 454.246);
        operators.l_Operator(239.05, 455.826);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(203.227, 452.524);
        operators.c_Operator(212.568, 451.636, 218.635, 438.239, 216.935, 429.609);
        operators.c_Operator(214.563, 417.571, 201.695, 413.687, 197.242, 403.464);
        operators.c_Operator(208.183, 418.703, 205.701, 443.45, 188.837, 452.883);
        operators.c_Operator(189.33, 454.311, 196.235, 453.948, 197.801, 453.502);
        operators.l_Operator(203.227, 452.524);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.5508, -20.1378, -20.1378, -3.5508, 205.1317, 431.0743);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(203.227, 452.524);
        operators.c_Operator(212.568, 451.636, 218.635, 438.239, 216.935, 429.609);
        operators.c_Operator(214.563, 417.571, 201.695, 413.687, 197.242, 403.464);
        operators.c_Operator(208.183, 418.703, 205.701, 443.45, 188.837, 452.883);
        operators.c_Operator(189.33, 454.311, 196.235, 453.948, 197.801, 453.502);
        operators.l_Operator(203.227, 452.524);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(159.812, 462.135);
        operators.c_Operator(163.632, 471.187, 168.637, 475.14, 178.055, 477.253);
        operators.c_Operator(188.513, 479.598, 200.069, 482.33, 207.881, 490.238);
        operators.c_Operator(203.858, 483.314, 199.762, 476.354, 195.772, 469.443);
        operators.c_Operator(193.05, 464.73, 190.384, 459.854, 185.315, 457.53);
        operators.c_Operator(176.595, 453.534, 166.501, 454.632, 158.673, 459.536);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.7844, -21.4625, -21.4625, -3.7844, 183.9779, 471.2473);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(159.812, 462.135);
        operators.c_Operator(163.632, 471.187, 168.637, 475.14, 178.055, 477.253);
        operators.c_Operator(188.513, 479.598, 200.069, 482.33, 207.881, 490.238);
        operators.c_Operator(203.858, 483.314, 199.762, 476.354, 195.772, 469.443);
        operators.c_Operator(193.05, 464.73, 190.384, 459.854, 185.315, 457.53);
        operators.c_Operator(176.595, 453.534, 166.501, 454.632, 158.673, 459.536);
        operators.S_Operator();
        painter.Save();
        operators.m_Operator(172.287, 448.175);
        operators.c_Operator(175.994, 448.305, 181.924, 443.139, 184.116, 440.257);
        operators.c_Operator(188.088, 435.036, 185.612, 428.354, 183.117, 422.971);
        operators.c_Operator(180.343, 416.991, 175.81, 412.19, 171.489, 407.37);
        operators.c_Operator(169.386, 405.022, 167.016, 401.909, 163.983, 400.811);
        operators.c_Operator(169.301, 408.847, 172.277, 422.283, 169.992, 431.841);
        operators.c_Operator(168.894, 436.441, 163.315, 450.019, 157.582, 449.978);
        operators.l_Operator(172.287, 448.175);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.5427, -20.0917, -20.0918, -3.5427, 173.2029, 427.5503);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(172.287, 448.175);
        operators.c_Operator(175.994, 448.305, 181.924, 443.139, 184.116, 440.257);
        operators.c_Operator(188.088, 435.036, 185.612, 428.354, 183.117, 422.971);
        operators.c_Operator(180.343, 416.991, 175.81, 412.19, 171.489, 407.37);
        operators.c_Operator(169.386, 405.022, 167.016, 401.909, 163.983, 400.811);
        operators.c_Operator(169.301, 408.847, 172.277, 422.283, 169.992, 431.841);
        operators.c_Operator(168.894, 436.441, 163.315, 450.019, 157.582, 449.978);
        operators.l_Operator(172.287, 448.175);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(162.793, 499.034);
        operators.c_Operator(160.29, 493.501, 161.148, 485.903, 160.168, 479.751);
        operators.c_Operator(159.19, 473.61, 155.142, 467.171, 149.975, 463.565);
        operators.c_Operator(145.037, 460.117, 135.738, 454.899, 130.138, 459.629);
        operators.c_Operator(125.373, 463.654, 126.898, 469.413, 130.794, 473.218);
        operators.c_Operator(140.169, 482.373, 157.057, 486.348, 162.793, 499.034);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.3929, -19.2418, -19.2419, -3.3929, 145.761, 477.6944);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(162.793, 499.034);
        operators.c_Operator(160.29, 493.501, 161.148, 485.903, 160.168, 479.751);
        operators.c_Operator(159.19, 473.61, 155.142, 467.171, 149.975, 463.565);
        operators.c_Operator(145.037, 460.117, 135.738, 454.899, 130.138, 459.629);
        operators.c_Operator(125.373, 463.654, 126.898, 469.413, 130.794, 473.218);
        operators.c_Operator(140.169, 482.373, 157.057, 486.348, 162.793, 499.034);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(218.71, 474.626);
        operators.c_Operator(227.977, 477.895, 241.395, 477.34, 247.686, 486.774);
        operators.c_Operator(241.59, 475.836, 237.621, 462.405, 225.582, 457.005);
        operators.c_Operator(220.354, 454.661, 207.091, 451.581, 202.287, 456.799);
        operators.c_Operator(196.622, 462.95, 214.129, 473.009, 218.71, 474.626);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.5005, -19.8525, -19.8526, -3.5005, 225.088, 469.4529);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(218.71, 474.626);
        operators.c_Operator(227.977, 477.895, 241.395, 477.34, 247.686, 486.774);
        operators.c_Operator(241.59, 475.836, 237.621, 462.405, 225.582, 457.005);
        operators.c_Operator(220.354, 454.661, 207.091, 451.581, 202.287, 456.799);
        operators.c_Operator(196.622, 462.95, 214.129, 473.009, 218.71, 474.626);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(364.219, 452.707);
        operators.c_Operator(372.337, 445.655, 386.137, 446.43, 394.847, 439.933);
        operators.c_Operator(398.901, 436.906, 401.746, 432.729, 403.427, 428.038);
        operators.c_Operator(405.013, 423.613, 403.627, 418.682, 405, 414.546);
        operators.c_Operator(409.187, 421.421, 410.231, 432.721, 401.178, 435.742);
        operators.c_Operator(393.654, 438.254, 387.016, 432.141, 379.632, 431.533);
        operators.c_Operator(373.305, 431.012, 364.799, 432.367, 359.767, 436.714);
        operators.c_Operator(356.337, 439.675, 354.885, 443.046, 354.845, 447.449);
        operators.c_Operator(354.826, 449.491, 356.647, 459.936, 358.087, 458.02);
        operators.l_Operator(364.219, 452.707);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.405, -24.9817, -24.9818, -4.405, 381.5099, 436.2673);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(364.219, 452.707);
        operators.c_Operator(372.337, 445.655, 386.137, 446.43, 394.847, 439.933);
        operators.c_Operator(398.901, 436.906, 401.746, 432.729, 403.427, 428.038);
        operators.c_Operator(405.013, 423.613, 403.627, 418.682, 405, 414.546);
        operators.c_Operator(409.187, 421.421, 410.231, 432.721, 401.178, 435.742);
        operators.c_Operator(393.654, 438.254, 387.016, 432.141, 379.632, 431.533);
        operators.c_Operator(373.305, 431.012, 364.799, 432.367, 359.767, 436.714);
        operators.c_Operator(356.337, 439.675, 354.885, 443.046, 354.845, 447.449);
        operators.c_Operator(354.826, 449.491, 356.647, 459.936, 358.087, 458.02);
        operators.l_Operator(364.219, 452.707);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(407.056, 462.106);
        operators.c_Operator(415.767, 463.399, 424.888, 467.415, 430.467, 474.275);
        operators.c_Operator(434.636, 479.402, 438.506, 485.855, 443.545, 488.941);
        operators.c_Operator(434.489, 483.237, 425.145, 477.969, 414.794, 475.07);
        operators.c_Operator(406.227, 472.672, 400.377, 469.951, 399.691, 460.808);
        operators.l_Operator(407.056, 462.106);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.2339, -18.3406, -18.3406, -3.2339, 421.5662, 474.8803);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(407.056, 462.106);
        operators.c_Operator(415.767, 463.399, 424.888, 467.415, 430.467, 474.275);
        operators.c_Operator(434.636, 479.402, 438.506, 485.855, 443.545, 488.941);
        operators.c_Operator(434.489, 483.237, 425.145, 477.969, 414.794, 475.07);
        operators.c_Operator(406.227, 472.672, 400.377, 469.951, 399.691, 460.808);
        operators.l_Operator(407.056, 462.106);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(326.54, 456.018);
        operators.c_Operator(331.199, 468.044, 339.261, 478.676, 353.363, 483.137);
        operators.c_Operator(365.438, 486.957, 377.709, 487.183, 383.053, 500.601);
        operators.c_Operator(381.962, 494.341, 379.622, 488.752, 375.136, 484.203);
        operators.c_Operator(369.883, 478.872, 363.998, 477.072, 357.412, 473.761);
        operators.c_Operator(352.87, 471.477, 349.706, 469.81, 348.133, 465.019);
        operators.c_Operator(347.402, 462.794, 349.723, 459.735, 341.763, 456.502);
        operators.y_Operator(333.803, 453.271, 326.54, 456.018);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.5353, -25.7208, -25.7208, -4.5353, 354.9188, 477.1113);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(326.54, 456.018);
        operators.c_Operator(331.199, 468.044, 339.261, 478.676, 353.363, 483.137);
        operators.c_Operator(365.438, 486.957, 377.709, 487.183, 383.053, 500.601);
        operators.c_Operator(381.962, 494.341, 379.622, 488.752, 375.136, 484.203);
        operators.c_Operator(369.883, 478.872, 363.998, 477.072, 357.412, 473.761);
        operators.c_Operator(352.87, 471.477, 349.706, 469.81, 348.133, 465.019);
        operators.c_Operator(347.402, 462.794, 349.723, 459.735, 341.763, 456.502);
        operators.y_Operator(333.803, 453.271, 326.54, 456.018);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(370.782, 474.492);
        operators.c_Operator(381.659, 476.41, 392.725, 479.692, 402.239, 488.626);
        operators.c_Operator(397.49, 476.273, 388.822, 462.157, 376.683, 456.29);
        operators.c_Operator(371.132, 453.607, 364.281, 451.645, 358.382, 454.895);
        operators.c_Operator(354.157, 457.222, 357.068, 463.893, 359.659, 468.469);
        operators.v_Operator(366.843, 473.798, 370.782, 474.492);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(7.7764, -20.7073, -20.7073, -7.7764, 377.9457, 470.3889);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(370.782, 474.492);
        operators.c_Operator(381.659, 476.41, 392.725, 479.692, 402.239, 488.626);
        operators.c_Operator(397.49, 476.273, 388.822, 462.157, 376.683, 456.29);
        operators.c_Operator(371.132, 453.607, 364.281, 451.645, 358.382, 454.895);
        operators.c_Operator(354.157, 457.222, 357.068, 463.893, 359.659, 468.469);
        operators.v_Operator(366.843, 473.798, 370.782, 474.492);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(393.954, 460.143);
        operators.c_Operator(417.7, 450.272, 419.674, 447.571, 423.723, 439.149);
        operators.y_Operator(426.794, 432.76, 427.888, 430.044);
        operators.c_Operator(428.376, 426.876, 430.27, 419.752, 427.137, 417.36);
        operators.c_Operator(424.288, 428.618, 411.909, 434.376, 401.127, 439.479);
        operators.c_Operator(396.819, 441.516, 390.324, 448.717, 391.924, 455.789);
        operators.l_Operator(393.954, 460.143);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(3.6459, -20.6767, -20.6767, -3.6459, 409.9052, 439.678);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(393.954, 460.143);
        operators.c_Operator(417.7, 450.272, 419.674, 447.571, 423.723, 439.149);
        operators.y_Operator(426.794, 432.76, 427.888, 430.044);
        operators.c_Operator(428.376, 426.876, 430.27, 419.752, 427.137, 417.36);
        operators.c_Operator(424.288, 428.618, 411.909, 434.376, 401.127, 439.479);
        operators.c_Operator(396.819, 441.516, 390.324, 448.717, 391.924, 455.789);
        operators.l_Operator(393.954, 460.143);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(422.5, 462.261);
        operators.c_Operator(425.645, 467.206, 431.813, 464.612, 435.358, 461.751);
        operators.c_Operator(440.883, 457.294, 443.278, 454.874, 450.737, 455.562);
        operators.c_Operator(445.896, 453.138, 436.797, 450.549, 431.699, 453.102);
        operators.c_Operator(427.056, 455.427, 427.184, 461.552, 423.071, 463.561);
        operators.l_Operator(422.5, 462.261);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(1.9037, -10.7962, -10.7962, -1.9037, 436.5559, 459.2133);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(422.5, 462.261);
        operators.c_Operator(425.645, 467.206, 431.813, 464.612, 435.358, 461.751);
        operators.c_Operator(440.883, 457.294, 443.278, 454.874, 450.737, 455.562);
        operators.c_Operator(445.896, 453.138, 436.797, 450.549, 431.699, 453.102);
        operators.c_Operator(427.056, 455.427, 427.184, 461.552, 423.071, 463.561);
        operators.l_Operator(422.5, 462.261);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(434.374, 461.577);
        operators.c_Operator(444.362, 464.432, 455.202, 466.667, 462.834, 456.814);
        operators.c_Operator(458.401, 459.789, 454.942, 459.762, 449.055, 459.609);
        operators.c_Operator(445.957, 459.528, 438.498, 456.856, 437.498, 460.363);
        operators.l_Operator(434.374, 461.577);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(1.6925, -9.5989, -9.5989, -1.6925, 449.5329, 459.4851);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(434.374, 461.577);
        operators.c_Operator(444.362, 464.432, 455.202, 466.667, 462.834, 456.814);
        operators.c_Operator(458.401, 459.789, 454.942, 459.762, 449.055, 459.609);
        operators.c_Operator(445.957, 459.528, 438.498, 456.856, 437.498, 460.363);
        operators.l_Operator(434.374, 461.577);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(426.533, 463.275);
        operators.c_Operator(429.489, 466.045, 435.765, 469.314, 439.459, 469.8);
        operators.c_Operator(443.975, 470.393, 448.299, 471.743, 452.264, 473.595);
        operators.c_Operator(451.065, 471.885, 437.205, 456.018, 429.009, 462.188);
        operators.l_Operator(426.533, 463.275);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(1.6254, -9.2183, -9.2184, -1.6254, 441.835, 467.0004);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(426.533, 463.275);
        operators.c_Operator(429.489, 466.045, 435.765, 469.314, 439.459, 469.8);
        operators.c_Operator(443.975, 470.393, 448.299, 471.743, 452.264, 473.595);
        operators.c_Operator(451.065, 471.885, 437.205, 456.018, 429.009, 462.188);
        operators.l_Operator(426.533, 463.275);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(411.42, 463.904);
        operators.c_Operator(417.99, 460.485, 424.876, 459.123, 430.923, 454.722);
        operators.c_Operator(434.164, 452.362, 441.6, 445.685, 439.435, 440.992);
        operators.c_Operator(436.306, 447.229, 428.165, 447.573, 423.514, 451.854);
        operators.c_Operator(420.577, 454.557, 418.315, 460.941, 414.543, 462.057);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(2.2998, -13.0429, -13.0429, -2.2998, 425.9059, 452.393);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(411.42, 463.904);
        operators.c_Operator(417.99, 460.485, 424.876, 459.123, 430.923, 454.722);
        operators.c_Operator(434.164, 452.362, 441.6, 445.685, 439.435, 440.992);
        operators.c_Operator(436.306, 447.229, 428.165, 447.573, 423.514, 451.854);
        operators.c_Operator(420.577, 454.557, 418.315, 460.941, 414.543, 462.057);
        operators.S_Operator();
        painter.Save();
        operators.m_Operator(142.023, 414.923);
        operators.c_Operator(132.833, 417.068, 123.392, 418.739, 114.447, 421.554);
        operators.c_Operator(102.458, 425.327, 89.808, 425.467, 82.914, 437.412);
        operators.c_Operator(77.371, 447.017, 77.455, 456.843, 89.978, 457.5);
        operators.c_Operator(114.013, 458.761, 120.904, 422.389, 142.023, 414.923);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.6944, -26.6232, -26.6232, -4.6944, 109.967, 435.2646);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(142.023, 414.923);
        operators.c_Operator(132.833, 417.068, 123.392, 418.739, 114.447, 421.554);
        operators.c_Operator(102.458, 425.327, 89.808, 425.467, 82.914, 437.412);
        operators.c_Operator(77.371, 447.017, 77.455, 456.843, 89.978, 457.5);
        operators.c_Operator(114.013, 458.761, 120.904, 422.389, 142.023, 414.923);
        operators.s_Operator();
        painter.Save();
        operators.m_Operator(88.628, 478.937);
        operators.c_Operator(105.699, 486.601, 109.115, 492.217, 115.583, 508.388);
        operators.c_Operator(112.811, 493.077, 108.739, 472.011, 94.9, 462.006);
        operators.c_Operator(88.843, 457.629, 67.661, 450.671, 66.89, 461.862);
        operators.c_Operator(66.223, 471.537, 82.135, 476.021, 88.628, 478.937);
        operators.W_Operator();
        operators.n_Operator();
        painter.Save();
        operators.cm_Operator(4.455, -25.2656, -25.2657, -4.455, 91.8979, 481.2943);
        painter.GraphicsState.SetShadingDictionary(*radialShading);
        painter.Restore();
        painter.Restore();
        operators.m_Operator(88.628, 478.937);
        operators.c_Operator(105.699, 486.601, 109.115, 492.217, 115.583, 508.388);
        operators.c_Operator(112.811, 493.077, 108.739, 472.011, 94.9, 462.006);
        operators.c_Operator(88.843, 457.629, 67.661, 450.671, 66.89, 461.862);
        operators.c_Operator(66.223, 471.537, 82.135, 476.021, 88.628, 478.937);
        operators.s_Operator();
        painter.Save();
        operators.i_Operator(1);
        operators.re_Operator(65, 510, 400, -116);
        operators.W_Operator();
        operators.n_Operator();
        operators.RG_Operator(1, 1, 1);
        operators.J_Operator(PdfLineCapStyle::Square);
        operators.w_Operator(1);
        operators.M_Operator(10);
        operators.m_Operator(72.795, 368.922);
        operators.c_Operator(98.211, 386.249, 89.756, 427.812, 68.128, 448.994);
        operators.c_Operator(65.32, 451.744, 54.206, 449.013, 55.091, 441.589);
        operators.c_Operator(58.1, 416.367, 43.788, 381.647, 72.794, 366.922);
        operators.BStar_Operator();
        painter.Restore();
    }

    doc.Save(TestUtils::GetTestOutputFilePath("TestType3RadialShading.pdf"), PdfSaveOptions::NoMetadataUpdate);
    TestUtils::AreFilesEqual(TestUtils::GetTestOutputFilePath("TestType3RadialShading.pdf"), TestUtils::GetTestInputFilePath("TestType3RadialShadingRef.pdf"));
}
