// Copyright (c) 1997-2004
// Utrecht University (The Netherlands),
// ETH Zurich (Switzerland),
// INRIA Sophia-Antipolis (France),
// Max-Planck-Institute Saarbruecken (Germany),
// and Tel-Aviv University (Israel).  All rights reserved.
//
// This file is part of CGAL (www.cgal.org)
//
// $URL: https://github.com/CGAL/cgal/blob/v6.1-beta2/Cartesian_kernel/include/CGAL/Cartesian/Circle_2.h $
// $Id: include/CGAL/Cartesian/Circle_2.h 335d5508930 $
// SPDX-License-Identifier: LGPL-3.0-or-later OR LicenseRef-Commercial
//
//
// Author(s)     : Andreas Fabri, Herve Bronnimann

#ifndef CGAL_CARTESIAN_CIRCLE_2_H
#define CGAL_CARTESIAN_CIRCLE_2_H

#include <CGAL/Cartesian/predicates_on_points_2.h>
#include <boost/tuple/tuple.hpp>

namespace CGAL {

template <class R_ >
class CircleC2
{
  typedef typename R_::Boolean              Boolean;
  typedef typename R_::FT                   FT;
  typedef typename R_::RT                   RT;
  typedef typename R_::Circle_2             Circle_2;
  typedef typename R_::Point_2              Point_2;

  typedef boost::tuple<Point_2, FT, Orientation>   Rep;
  typedef typename R_::template Handle<Rep>::type  Base;

  Base base;

public:
  typedef R_                                     R;

  CircleC2() {}

  explicit CircleC2(const Point_2 &center, const FT &squared_radius = FT(0),
           const Orientation &orient = COUNTERCLOCKWISE) // Is this new?
  {
    CGAL_kernel_precondition( ( squared_radius >= FT(0) ) &
                              ( orient    != COLLINEAR) );

    base = Rep(center, squared_radius, orient);
  }

  Boolean operator==(const CircleC2& s) const;
  Boolean operator!=(const CircleC2& s) const;

  const Point_2 & center() const
  {
    return get_pointee_or_identity(base).template get<0>();
  }

  const FT & squared_radius() const
  {
    return get_pointee_or_identity(base).template get<1>();
  }

  Orientation orientation() const
  {
    return get_pointee_or_identity(base).template get<2>();
  }

};

template < class R >
typename R::Boolean
CircleC2<R>::operator==(const CircleC2<R> &t) const
{
  if (CGAL::identical(base, t.base))
    return true;

  return center() == t.center() &&
         squared_radius() == t.squared_radius() &&
         orientation() == t.orientation();
}

template < class R >
typename R::Boolean
CircleC2<R>::operator!=(const CircleC2<R> &t) const
{
  return !(*this == t);
}

} //namespace CGAL

#endif // CGAL_CARTESIAN_CIRCLE_2_H
