// clang-format off
/* ----------------------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   LAMMPS development team: developers@lammps.org

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#include "atom_vec_molecular_kokkos.h"

#include "atom_kokkos.h"
#include "atom_masks.h"
#include "comm_kokkos.h"
#include "domain.h"
#include "error.h"
#include "fix.h"
#include "memory_kokkos.h"
#include "modify.h"

using namespace LAMMPS_NS;

/* ---------------------------------------------------------------------- */

AtomVecMolecularKokkos::AtomVecMolecularKokkos(LAMMPS *lmp) : AtomVec(lmp),
AtomVecKokkos(lmp), AtomVecMolecular(lmp)
{
  unpack_exchange_indices_flag = 1;
}

/* ----------------------------------------------------------------------
   grow atom arrays
   n = 0 grows arrays by DELTA
   n > 0 allocates arrays to size n
------------------------------------------------------------------------- */

void AtomVecMolecularKokkos::grow(int n)
{
  auto DELTA = LMP_KOKKOS_AV_DELTA;
  int step = MAX(DELTA,nmax*0.01);
  if (n == 0) nmax += step;
  else nmax = n;
  atomKK->nmax = nmax;
  if (nmax < 0 || nmax > MAXSMALLINT)
    error->one(FLERR,"Per-processor system is too big");

  atomKK->sync(Device,ALL_MASK);
  atomKK->modified(Device,ALL_MASK);

  memoryKK->grow_kokkos(atomKK->k_tag,atomKK->tag,nmax,"atom:tag");
  memoryKK->grow_kokkos(atomKK->k_type,atomKK->type,nmax,"atom:type");
  memoryKK->grow_kokkos(atomKK->k_mask,atomKK->mask,nmax,"atom:mask");
  memoryKK->grow_kokkos(atomKK->k_image,atomKK->image,nmax,"atom:image");

  memoryKK->grow_kokkos(atomKK->k_x,atomKK->x,nmax,"atom:x");
  memoryKK->grow_kokkos(atomKK->k_v,atomKK->v,nmax,"atom:v");
  memoryKK->grow_kokkos(atomKK->k_f,atomKK->f,nmax,"atom:f");

  memoryKK->grow_kokkos(atomKK->k_molecule,atomKK->molecule,nmax,"atom:molecule");
  memoryKK->grow_kokkos(atomKK->k_nspecial,atomKK->nspecial,nmax,3,"atom:nspecial");
  memoryKK->grow_kokkos(atomKK->k_special,atomKK->special,nmax,atomKK->maxspecial,
                      "atom:special");
  memoryKK->grow_kokkos(atomKK->k_num_bond,atomKK->num_bond,nmax,"atom:num_bond");
  memoryKK->grow_kokkos(atomKK->k_bond_type,atomKK->bond_type,nmax,atomKK->bond_per_atom,
                      "atom:bond_type");
  memoryKK->grow_kokkos(atomKK->k_bond_atom,atomKK->bond_atom,nmax,atomKK->bond_per_atom,
                      "atom:bond_atom");

  memoryKK->grow_kokkos(atomKK->k_num_angle,atomKK->num_angle,nmax,"atom:num_angle");
  memoryKK->grow_kokkos(atomKK->k_angle_type,atomKK->angle_type,nmax,atomKK->angle_per_atom,
                      "atom:angle_type");
  memoryKK->grow_kokkos(atomKK->k_angle_atom1,atomKK->angle_atom1,nmax,atomKK->angle_per_atom,
                      "atom:angle_atom1");
  memoryKK->grow_kokkos(atomKK->k_angle_atom2,atomKK->angle_atom2,nmax,atomKK->angle_per_atom,
                      "atom:angle_atom2");
  memoryKK->grow_kokkos(atomKK->k_angle_atom3,atomKK->angle_atom3,nmax,atomKK->angle_per_atom,
                      "atom:angle_atom3");

  memoryKK->grow_kokkos(atomKK->k_num_dihedral,atomKK->num_dihedral,nmax,"atom:num_dihedral");
  memoryKK->grow_kokkos(atomKK->k_dihedral_type,atomKK->dihedral_type,nmax,
                      atomKK->dihedral_per_atom,"atom:dihedral_type");
  memoryKK->grow_kokkos(atomKK->k_dihedral_atom1,atomKK->dihedral_atom1,nmax,
                      atomKK->dihedral_per_atom,"atom:dihedral_atom1");
  memoryKK->grow_kokkos(atomKK->k_dihedral_atom2,atomKK->dihedral_atom2,nmax,
                      atomKK->dihedral_per_atom,"atom:dihedral_atom2");
  memoryKK->grow_kokkos(atomKK->k_dihedral_atom3,atomKK->dihedral_atom3,nmax,
                      atomKK->dihedral_per_atom,"atom:dihedral_atom3");
  memoryKK->grow_kokkos(atomKK->k_dihedral_atom4,atomKK->dihedral_atom4,nmax,
                      atomKK->dihedral_per_atom,"atom:dihedral_atom4");

  memoryKK->grow_kokkos(atomKK->k_num_improper,atomKK->num_improper,nmax,"atom:num_improper");
  memoryKK->grow_kokkos(atomKK->k_improper_type,atomKK->improper_type,nmax,
                      atomKK->improper_per_atom,"atom:improper_type");
  memoryKK->grow_kokkos(atomKK->k_improper_atom1,atomKK->improper_atom1,nmax,
                      atomKK->improper_per_atom,"atom:improper_atom1");
  memoryKK->grow_kokkos(atomKK->k_improper_atom2,atomKK->improper_atom2,nmax,
                      atomKK->improper_per_atom,"atom:improper_atom2");
  memoryKK->grow_kokkos(atomKK->k_improper_atom3,atomKK->improper_atom3,nmax,
                      atomKK->improper_per_atom,"atom:improper_atom3");
  memoryKK->grow_kokkos(atomKK->k_improper_atom4,atomKK->improper_atom4,nmax,
                      atomKK->improper_per_atom,"atom:improper_atom4");

  grow_pointers();
  atomKK->sync(Host,ALL_MASK);

  if (atom->nextra_grow)
    for (int iextra = 0; iextra < atom->nextra_grow; iextra++)
      modify->fix[atom->extra_grow[iextra]]->grow_arrays(nmax);
}

/* ----------------------------------------------------------------------
   reset local array ptrs
------------------------------------------------------------------------- */

void AtomVecMolecularKokkos::grow_pointers()
{
  tag = atomKK->tag;
  d_tag = atomKK->k_tag.d_view;
  h_tag = atomKK->k_tag.h_view;

  type = atomKK->type;
  d_type = atomKK->k_type.d_view;
  h_type = atomKK->k_type.h_view;
  mask = atomKK->mask;
  d_mask = atomKK->k_mask.d_view;
  h_mask = atomKK->k_mask.h_view;
  image = atomKK->image;
  d_image = atomKK->k_image.d_view;
  h_image = atomKK->k_image.h_view;

  x = atomKK->x;
  d_x = atomKK->k_x.d_view;
  h_x = atomKK->k_x.h_viewkk;
  v = atomKK->v;
  d_v = atomKK->k_v.d_view;
  h_v = atomKK->k_v.h_viewkk;
  f = atomKK->f;
  d_f = atomKK->k_f.d_view;
  h_f = atomKK->k_f.h_viewkk;

  molecule = atomKK->molecule;
  d_molecule = atomKK->k_molecule.d_view;
  h_molecule = atomKK->k_molecule.h_view;
  nspecial = atomKK->nspecial;
  d_nspecial = atomKK->k_nspecial.d_view;
  h_nspecial = atomKK->k_nspecial.h_viewkk;
  special = atomKK->special;
  d_special = atomKK->k_special.d_view;
  h_special = atomKK->k_special.h_viewkk;
  num_bond = atomKK->num_bond;
  d_num_bond = atomKK->k_num_bond.d_view;
  h_num_bond = atomKK->k_num_bond.h_view;
  bond_type = atomKK->bond_type;
  d_bond_type = atomKK->k_bond_type.d_view;
  h_bond_type = atomKK->k_bond_type.h_viewkk;
  bond_atom = atomKK->bond_atom;
  d_bond_atom = atomKK->k_bond_atom.d_view;
  h_bond_atom = atomKK->k_bond_atom.h_viewkk;

  num_angle = atomKK->num_angle;
  d_num_angle = atomKK->k_num_angle.d_view;
  h_num_angle = atomKK->k_num_angle.h_view;
  angle_type = atomKK->angle_type;
  d_angle_type = atomKK->k_angle_type.d_view;
  h_angle_type = atomKK->k_angle_type.h_viewkk;
  angle_atom1 = atomKK->angle_atom1;
  d_angle_atom1 = atomKK->k_angle_atom1.d_view;
  h_angle_atom1 = atomKK->k_angle_atom1.h_viewkk;
  angle_atom2 = atomKK->angle_atom2;
  d_angle_atom2 = atomKK->k_angle_atom2.d_view;
  h_angle_atom2 = atomKK->k_angle_atom2.h_viewkk;
  angle_atom3 = atomKK->angle_atom3;
  d_angle_atom3 = atomKK->k_angle_atom3.d_view;
  h_angle_atom3 = atomKK->k_angle_atom3.h_viewkk;

  num_dihedral = atomKK->num_dihedral;
  d_num_dihedral = atomKK->k_num_dihedral.d_view;
  h_num_dihedral = atomKK->k_num_dihedral.h_view;
  dihedral_type = atomKK->dihedral_type;
  d_dihedral_type = atomKK->k_dihedral_type.d_view;
  h_dihedral_type = atomKK->k_dihedral_type.h_viewkk;
  dihedral_atom1 = atomKK->dihedral_atom1;
  d_dihedral_atom1 = atomKK->k_dihedral_atom1.d_view;
  h_dihedral_atom1 = atomKK->k_dihedral_atom1.h_viewkk;
  dihedral_atom2 = atomKK->dihedral_atom2;
  d_dihedral_atom2 = atomKK->k_dihedral_atom2.d_view;
  h_dihedral_atom2 = atomKK->k_dihedral_atom2.h_viewkk;
  dihedral_atom3 = atomKK->dihedral_atom3;
  d_dihedral_atom3 = atomKK->k_dihedral_atom3.d_view;
  h_dihedral_atom3 = atomKK->k_dihedral_atom3.h_viewkk;
  dihedral_atom4 = atomKK->dihedral_atom4;
  d_dihedral_atom4 = atomKK->k_dihedral_atom4.d_view;
  h_dihedral_atom4 = atomKK->k_dihedral_atom4.h_viewkk;

  num_improper = atomKK->num_improper;
  d_num_improper = atomKK->k_num_improper.d_view;
  h_num_improper = atomKK->k_num_improper.h_view;
  improper_type = atomKK->improper_type;
  d_improper_type = atomKK->k_improper_type.d_view;
  h_improper_type = atomKK->k_improper_type.h_viewkk;
  improper_atom1 = atomKK->improper_atom1;
  d_improper_atom1 = atomKK->k_improper_atom1.d_view;
  h_improper_atom1 = atomKK->k_improper_atom1.h_viewkk;
  improper_atom2 = atomKK->improper_atom2;
  d_improper_atom2 = atomKK->k_improper_atom2.d_view;
  h_improper_atom2 = atomKK->k_improper_atom2.h_viewkk;
  improper_atom3 = atomKK->improper_atom3;
  d_improper_atom3 = atomKK->k_improper_atom3.d_view;
  h_improper_atom3 = atomKK->k_improper_atom3.h_viewkk;
  improper_atom4 = atomKK->improper_atom4;
  d_improper_atom4 = atomKK->k_improper_atom4.d_view;
  h_improper_atom4 = atomKK->k_improper_atom4.h_viewkk;
}

/* ----------------------------------------------------------------------
   sort atom arrays on device
------------------------------------------------------------------------- */

void AtomVecMolecularKokkos::sort_kokkos(Kokkos::BinSort<KeyViewType, BinOp> &Sorter)
{
  atomKK->sync(Device, ALL_MASK & ~F_MASK);

  Sorter.sort(LMPDeviceType(), d_tag);
  Sorter.sort(LMPDeviceType(), d_type);
  Sorter.sort(LMPDeviceType(), d_mask);
  Sorter.sort(LMPDeviceType(), d_image);
  Sorter.sort(LMPDeviceType(), d_x);
  Sorter.sort(LMPDeviceType(), d_v);
  Sorter.sort(LMPDeviceType(), d_molecule);
  Sorter.sort(LMPDeviceType(), d_num_bond);
  Sorter.sort(LMPDeviceType(), d_bond_type);
  Sorter.sort(LMPDeviceType(), d_bond_atom);
  Sorter.sort(LMPDeviceType(), d_nspecial);
  Sorter.sort(LMPDeviceType(), d_special);
  Sorter.sort(LMPDeviceType(), d_num_angle);
  Sorter.sort(LMPDeviceType(), d_angle_type);
  Sorter.sort(LMPDeviceType(), d_angle_atom1);
  Sorter.sort(LMPDeviceType(), d_angle_atom2);
  Sorter.sort(LMPDeviceType(), d_angle_atom3);
  Sorter.sort(LMPDeviceType(), d_num_dihedral);
  Sorter.sort(LMPDeviceType(), d_dihedral_type);
  Sorter.sort(LMPDeviceType(), d_dihedral_atom1);
  Sorter.sort(LMPDeviceType(), d_dihedral_atom2);
  Sorter.sort(LMPDeviceType(), d_dihedral_atom3);
  Sorter.sort(LMPDeviceType(), d_dihedral_atom4);
  Sorter.sort(LMPDeviceType(), d_num_improper);
  Sorter.sort(LMPDeviceType(), d_improper_type);
  Sorter.sort(LMPDeviceType(), d_improper_atom1);
  Sorter.sort(LMPDeviceType(), d_improper_atom2);
  Sorter.sort(LMPDeviceType(), d_improper_atom3);
  Sorter.sort(LMPDeviceType(), d_improper_atom4);

  atomKK->modified(Device, ALL_MASK & ~F_MASK);
}

/* ---------------------------------------------------------------------- */

template<class DeviceType,int PBC_FLAG>
struct AtomVecMolecularKokkos_PackBorder {
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;

  typename AT::t_double_2d_lr _buf;
  const typename AT::t_int_1d_const _list;
  const typename AT::t_kkfloat_1d_3_lr_randomread _x;
  const typename AT::t_tagint_1d _tag;
  const typename AT::t_int_1d _type;
  const typename AT::t_int_1d _mask;
  const typename AT::t_tagint_1d _molecule;
  double _dx,_dy,_dz;

  AtomVecMolecularKokkos_PackBorder(
      const typename AT::t_double_2d_lr &buf,
      const typename AT::t_int_1d_const &list,
      const typename AT::t_kkfloat_1d_3_lr &x,
      const typename AT::t_tagint_1d &tag,
      const typename AT::t_int_1d &type,
      const typename AT::t_int_1d &mask,
      const typename AT::t_tagint_1d &molecule,
      const double &dx, const double &dy, const double &dz):
      _buf(buf),_list(list),
      _x(x),_tag(tag),_type(type),_mask(mask),_molecule(molecule),
      _dx(dx),_dy(dy),_dz(dz) {}

  KOKKOS_INLINE_FUNCTION
  void operator() (const int& i) const {
      const int j = _list(i);
      if (PBC_FLAG == 0) {
          _buf(i,0) = _x(j,0);
          _buf(i,1) = _x(j,1);
          _buf(i,2) = _x(j,2);
          _buf(i,3) = d_ubuf(_tag(j)).d;
          _buf(i,4) = d_ubuf(_type(j)).d;
          _buf(i,5) = d_ubuf(_mask(j)).d;
          _buf(i,6) = d_ubuf(_molecule(j)).d;
      } else {
          _buf(i,0) = _x(j,0) + _dx;
          _buf(i,1) = _x(j,1) + _dy;
          _buf(i,2) = _x(j,2) + _dz;
          _buf(i,3) = d_ubuf(_tag(j)).d;
          _buf(i,4) = d_ubuf(_type(j)).d;
          _buf(i,5) = d_ubuf(_mask(j)).d;
          _buf(i,6) = d_ubuf(_molecule(j)).d;
      }
  }
};

/* ---------------------------------------------------------------------- */

int AtomVecMolecularKokkos::pack_border_kokkos(int n, DAT::tdual_int_1d k_sendlist,
                                               DAT::tdual_double_2d_lr buf,
                                               int pbc_flag, int *pbc, ExecutionSpace space)
{
  double dx,dy,dz;

  if (pbc_flag != 0) {
    if (domain->triclinic == 0) {
      dx = pbc[0]*domain->xprd;
      dy = pbc[1]*domain->yprd;
      dz = pbc[2]*domain->zprd;
    } else {
      dx = pbc[0];
      dy = pbc[1];
      dz = pbc[2];
    }
    if (space==Host) {
      AtomVecMolecularKokkos_PackBorder<LMPHostType,1> f(
        buf.h_view, k_sendlist.h_view,
        h_x,h_tag,h_type,h_mask,h_molecule,dx,dy,dz);
      Kokkos::parallel_for(n,f);
    } else {
      AtomVecMolecularKokkos_PackBorder<LMPDeviceType,1> f(
        buf.d_view, k_sendlist.d_view,
        d_x,d_tag,d_type,d_mask,d_molecule,dx,dy,dz);
      Kokkos::parallel_for(n,f);
    }

  } else {
    dx = dy = dz = 0;
    if (space==Host) {
      AtomVecMolecularKokkos_PackBorder<LMPHostType,0> f(
        buf.h_view, k_sendlist.h_view,
        h_x,h_tag,h_type,h_mask,h_molecule,dx,dy,dz);
      Kokkos::parallel_for(n,f);
    } else {
      AtomVecMolecularKokkos_PackBorder<LMPDeviceType,0> f(
        buf.d_view, k_sendlist.d_view,
        d_x,d_tag,d_type,d_mask,d_molecule,dx,dy,dz);
      Kokkos::parallel_for(n,f);
    }
  }
  return n*size_border;
}

/* ---------------------------------------------------------------------- */

template<class DeviceType>
struct AtomVecMolecularKokkos_UnpackBorder {
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;

  const typename AT::t_double_2d_lr_const _buf;
  typename AT::t_kkfloat_1d_3_lr _x;
  typename AT::t_tagint_1d _tag;
  typename AT::t_int_1d _type;
  typename AT::t_int_1d _mask;
  typename AT::t_tagint_1d _molecule;
  int _first;


  AtomVecMolecularKokkos_UnpackBorder(
      const typename AT::t_double_2d_lr_const &buf,
      typename AT::t_kkfloat_1d_3_lr &x,
      typename AT::t_tagint_1d &tag,
      typename AT::t_int_1d &type,
      typename AT::t_int_1d &mask,
      typename AT::t_tagint_1d &molecule,
      const int& first):
    _buf(buf),_x(x),_tag(tag),_type(type),_mask(mask),_molecule(molecule),
    _first(first) {
  };

  KOKKOS_INLINE_FUNCTION
  void operator() (const int& i) const {
      _x(i+_first,0) = _buf(i,0);
      _x(i+_first,1) = _buf(i,1);
      _x(i+_first,2) = _buf(i,2);
      _tag(i+_first) = (tagint) d_ubuf(_buf(i,3)).i;
      _type(i+_first) = (int) d_ubuf(_buf(i,4)).i;
      _mask(i+_first) = (int) d_ubuf(_buf(i,5)).i;
      _molecule(i+_first) = (tagint) d_ubuf(_buf(i,6)).i;

  }
};

/* ---------------------------------------------------------------------- */

void AtomVecMolecularKokkos::unpack_border_kokkos(const int &n, const int &first,
                                                  const DAT::tdual_double_2d_lr &buf,
                                                  ExecutionSpace space) {
  atomKK->modified(space,X_MASK|TAG_MASK|TYPE_MASK|MASK_MASK|MOLECULE_MASK);
  while (first+n >= nmax) grow(0);
  atomKK->modified(space,X_MASK|TAG_MASK|TYPE_MASK|MASK_MASK|MOLECULE_MASK);
  if (space==Host) {
    struct AtomVecMolecularKokkos_UnpackBorder<LMPHostType>
      f(buf.h_view,h_x,h_tag,h_type,h_mask,h_molecule,first);
    Kokkos::parallel_for(n,f);
  } else {
    struct AtomVecMolecularKokkos_UnpackBorder<LMPDeviceType>
      f(buf.d_view,d_x,d_tag,d_type,d_mask,d_molecule,first);
    Kokkos::parallel_for(n,f);
  }
}

/* ---------------------------------------------------------------------- */

template<class DeviceType>
struct AtomVecMolecularKokkos_PackExchangeFunctor {
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;
  typename AT::t_kkfloat_1d_3_lr_randomread _x;
  typename AT::t_kkfloat_1d_3_randomread _v;
  typename AT::t_tagint_1d_randomread _tag;
  typename AT::t_int_1d_randomread _type;
  typename AT::t_int_1d_randomread _mask;
  typename AT::t_imageint_1d_randomread _image;
  typename AT::t_tagint_1d_randomread _molecule;
  typename AT::t_int_2d_randomread _nspecial;
  typename AT::t_tagint_2d_randomread _special;
  typename AT::t_int_1d_randomread _num_bond;
  typename AT::t_int_2d_randomread _bond_type;
  typename AT::t_tagint_2d_randomread _bond_atom;
  typename AT::t_int_1d_randomread _num_angle;
  typename AT::t_int_2d_randomread _angle_type;
  typename AT::t_tagint_2d_randomread _angle_atom1,_angle_atom2,_angle_atom3;
  typename AT::t_int_1d_randomread _num_dihedral;
  typename AT::t_int_2d_randomread _dihedral_type;
  typename AT::t_tagint_2d_randomread _dihedral_atom1,_dihedral_atom2,
    _dihedral_atom3,_dihedral_atom4;
  typename AT::t_int_1d_randomread _num_improper;
  typename AT::t_int_2d_randomread _improper_type;
  typename AT::t_tagint_2d_randomread _improper_atom1,_improper_atom2,
    _improper_atom3,_improper_atom4;
  typename AT::t_kkfloat_1d_3_lr _xw;
  typename AT::t_kkfloat_1d_3 _vw;
  typename AT::t_tagint_1d _tagw;
  typename AT::t_int_1d _typew;
  typename AT::t_int_1d _maskw;
  typename AT::t_imageint_1d _imagew;
  typename AT::t_tagint_1d _moleculew;
  typename AT::t_int_2d _nspecialw;
  typename AT::t_tagint_2d _specialw;
  typename AT::t_int_1d _num_bondw;
  typename AT::t_int_2d _bond_typew;
  typename AT::t_tagint_2d _bond_atomw;
  typename AT::t_int_1d _num_anglew;
  typename AT::t_int_2d _angle_typew;
  typename AT::t_tagint_2d _angle_atom1w,_angle_atom2w,_angle_atom3w;
  typename AT::t_int_1d _num_dihedralw;
  typename AT::t_int_2d _dihedral_typew;
  typename AT::t_tagint_2d _dihedral_atom1w,_dihedral_atom2w,
    _dihedral_atom3w,_dihedral_atom4w;
  typename AT::t_int_1d _num_improperw;
  typename AT::t_int_2d _improper_typew;
  typename AT::t_tagint_2d _improper_atom1w,_improper_atom2w,
    _improper_atom3w,_improper_atom4w;
  typename AT::t_double_2d_lr_um _buf;
  typename AT::t_int_1d_const _sendlist;
  typename AT::t_int_1d_const _copylist;
  int _size_exchange;

  AtomVecMolecularKokkos_PackExchangeFunctor(
    const AtomKokkos* atom,
    const DAT::tdual_double_2d_lr buf,
    DAT::tdual_int_1d sendlist,
    DAT::tdual_int_1d copylist):
      _x(atom->k_x.view<DeviceType>()),
      _v(atom->k_v.view<DeviceType>()),
      _tag(atom->k_tag.view<DeviceType>()),
      _type(atom->k_type.view<DeviceType>()),
      _mask(atom->k_mask.view<DeviceType>()),
      _image(atom->k_image.view<DeviceType>()),
      _molecule(atom->k_molecule.view<DeviceType>()),
      _nspecial(atom->k_nspecial.view<DeviceType>()),
      _special(atom->k_special.view<DeviceType>()),
      _num_bond(atom->k_num_bond.view<DeviceType>()),
      _bond_type(atom->k_bond_type.view<DeviceType>()),
      _bond_atom(atom->k_bond_atom.view<DeviceType>()),
      _num_angle(atom->k_num_angle.view<DeviceType>()),
      _angle_type(atom->k_angle_type.view<DeviceType>()),
      _angle_atom1(atom->k_angle_atom1.view<DeviceType>()),
      _angle_atom2(atom->k_angle_atom2.view<DeviceType>()),
      _angle_atom3(atom->k_angle_atom3.view<DeviceType>()),
      _num_dihedral(atom->k_num_dihedral.view<DeviceType>()),
      _dihedral_type(atom->k_dihedral_type.view<DeviceType>()),
      _dihedral_atom1(atom->k_dihedral_atom1.view<DeviceType>()),
      _dihedral_atom2(atom->k_dihedral_atom2.view<DeviceType>()),
      _dihedral_atom3(atom->k_dihedral_atom3.view<DeviceType>()),
      _dihedral_atom4(atom->k_dihedral_atom4.view<DeviceType>()),
      _num_improper(atom->k_num_improper.view<DeviceType>()),
      _improper_type(atom->k_improper_type.view<DeviceType>()),
      _improper_atom1(atom->k_improper_atom1.view<DeviceType>()),
      _improper_atom2(atom->k_improper_atom2.view<DeviceType>()),
      _improper_atom3(atom->k_improper_atom3.view<DeviceType>()),
      _improper_atom4(atom->k_improper_atom4.view<DeviceType>()),
      _xw(atom->k_x.view<DeviceType>()),
      _vw(atom->k_v.view<DeviceType>()),
      _tagw(atom->k_tag.view<DeviceType>()),
      _typew(atom->k_type.view<DeviceType>()),
      _maskw(atom->k_mask.view<DeviceType>()),
      _imagew(atom->k_image.view<DeviceType>()),
      _moleculew(atom->k_molecule.view<DeviceType>()),
      _nspecialw(atom->k_nspecial.view<DeviceType>()),
      _specialw(atom->k_special.view<DeviceType>()),
      _num_bondw(atom->k_num_bond.view<DeviceType>()),
      _bond_typew(atom->k_bond_type.view<DeviceType>()),
      _bond_atomw(atom->k_bond_atom.view<DeviceType>()),
      _num_anglew(atom->k_num_angle.view<DeviceType>()),
      _angle_typew(atom->k_angle_type.view<DeviceType>()),
      _angle_atom1w(atom->k_angle_atom1.view<DeviceType>()),
      _angle_atom2w(atom->k_angle_atom2.view<DeviceType>()),
      _angle_atom3w(atom->k_angle_atom3.view<DeviceType>()),
      _num_dihedralw(atom->k_num_dihedral.view<DeviceType>()),
      _dihedral_typew(atom->k_dihedral_type.view<DeviceType>()),
      _dihedral_atom1w(atom->k_dihedral_atom1.view<DeviceType>()),
      _dihedral_atom2w(atom->k_dihedral_atom2.view<DeviceType>()),
      _dihedral_atom3w(atom->k_dihedral_atom3.view<DeviceType>()),
      _dihedral_atom4w(atom->k_dihedral_atom4.view<DeviceType>()),
      _num_improperw(atom->k_num_improper.view<DeviceType>()),
      _improper_typew(atom->k_improper_type.view<DeviceType>()),
      _improper_atom1w(atom->k_improper_atom1.view<DeviceType>()),
      _improper_atom2w(atom->k_improper_atom2.view<DeviceType>()),
      _improper_atom3w(atom->k_improper_atom3.view<DeviceType>()),
      _improper_atom4w(atom->k_improper_atom4.view<DeviceType>()),
      _sendlist(sendlist.template view<DeviceType>()),
      _copylist(copylist.template view<DeviceType>()),
      _size_exchange(atom->avecKK->size_exchange) {
    const int maxsendlist = (buf.template view<DeviceType>().extent(0)*
                             buf.template view<DeviceType>().extent(1))/_size_exchange;
    buffer_view<DeviceType>(_buf,buf,maxsendlist,_size_exchange);
  }

  KOKKOS_INLINE_FUNCTION
  void operator() (const int &mysend) const {
    int k;
    const int i = _sendlist(mysend);
    _buf(mysend,0) = _size_exchange;
    int m = 1;
    _buf(mysend,m++) = _x(i,0);
    _buf(mysend,m++) = _x(i,1);
    _buf(mysend,m++) = _x(i,2);
    _buf(mysend,m++) = _v(i,0);
    _buf(mysend,m++) = _v(i,1);
    _buf(mysend,m++) = _v(i,2);
    _buf(mysend,m++) = d_ubuf(_tag(i)).d;
    _buf(mysend,m++) = d_ubuf(_type(i)).d;
    _buf(mysend,m++) = d_ubuf(_mask(i)).d;
    _buf(mysend,m++) = d_ubuf(_image(i)).d;
    _buf(mysend,m++) = d_ubuf(_molecule(i)).d;
    _buf(mysend,m++) = d_ubuf(_num_bond(i)).d;
    for (k = 0; k < _num_bond(i); k++) {
      _buf(mysend,m++) = d_ubuf(_bond_type(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_bond_atom(i,k)).d;
    }
    _buf(mysend,m++) = d_ubuf(_num_angle(i)).d;
    for (k = 0; k < _num_angle(i); k++) {
      _buf(mysend,m++) = d_ubuf(_angle_type(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_angle_atom1(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_angle_atom2(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_angle_atom3(i,k)).d;
    }
    _buf(mysend,m++) = d_ubuf(_num_dihedral(i)).d;
    for (k = 0; k < _num_dihedral(i); k++) {
      _buf(mysend,m++) = d_ubuf(_dihedral_type(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_dihedral_atom1(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_dihedral_atom2(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_dihedral_atom3(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_dihedral_atom4(i,k)).d;
    }
    _buf(mysend,m++) = d_ubuf(_num_improper(i)).d;
    for (k = 0; k < _num_improper(i); k++) {
      _buf(mysend,m++) = d_ubuf(_improper_type(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_improper_atom1(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_improper_atom2(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_improper_atom3(i,k)).d;
      _buf(mysend,m++) = d_ubuf(_improper_atom4(i,k)).d;
    }

    _buf(mysend,m++) = d_ubuf(_nspecial(i,0)).d;
    _buf(mysend,m++) = d_ubuf(_nspecial(i,1)).d;
    _buf(mysend,m++) = d_ubuf(_nspecial(i,2)).d;
    for (k = 0; k < _nspecial(i,2); k++)
      _buf(mysend,m++) = d_ubuf(_special(i,k)).d;

    const int j = _copylist(mysend);

    if (j>-1) {
    _xw(i,0) = _x(j,0);
    _xw(i,1) = _x(j,1);
    _xw(i,2) = _x(j,2);
    _vw(i,0) = _v(j,0);
    _vw(i,1) = _v(j,1);
    _vw(i,2) = _v(j,2);
    _tagw(i) = _tag(j);
    _typew(i) = _type(j);
    _maskw(i) = _mask(j);
    _imagew(i) = _image(j);
    _moleculew(i) = _molecule(j);
    _num_bondw(i) = _num_bond(j);
    for (k = 0; k < _num_bond(j); k++) {
      _bond_typew(i,k) = _bond_type(j,k);
      _bond_atomw(i,k) = _bond_atom(j,k);
    }
    _num_anglew(i) = _num_angle(j);
    for (k = 0; k < _num_angle(j); k++) {
      _angle_typew(i,k) = _angle_type(j,k);
      _angle_atom1w(i,k) = _angle_atom1(j,k);
      _angle_atom2w(i,k) = _angle_atom2(j,k);
      _angle_atom3w(i,k) = _angle_atom3(j,k);
    }
    _num_dihedralw(i) = _num_dihedral(j);
    for (k = 0; k < _num_dihedral(j); k++) {
      _dihedral_typew(i,k) = _dihedral_type(j,k);
      _dihedral_atom1w(i,k) = _dihedral_atom1(j,k);
      _dihedral_atom2w(i,k) = _dihedral_atom2(j,k);
      _dihedral_atom3w(i,k) = _dihedral_atom3(j,k);
      _dihedral_atom4w(i,k) = _dihedral_atom4(j,k);
    }
    _num_improperw(i) = _num_improper(j);
    for (k = 0; k < _num_improper(j); k++) {
      _improper_typew(i,k) = _improper_type(j,k);
      _improper_atom1w(i,k) = _improper_atom1(j,k);
      _improper_atom2w(i,k) = _improper_atom2(j,k);
      _improper_atom3w(i,k) = _improper_atom3(j,k);
      _improper_atom4w(i,k) = _improper_atom4(j,k);
    }
    _nspecialw(i,0) = _nspecial(j,0);
    _nspecialw(i,1) = _nspecial(j,1);
    _nspecialw(i,2) = _nspecial(j,2);
    for (k = 0; k < _nspecial(j,2); k++)
      _specialw(i,k) = _special(j,k);
    }
  }
};

/* ---------------------------------------------------------------------- */

int AtomVecMolecularKokkos::pack_exchange_kokkos(const int &nsend,DAT::tdual_double_2d_lr &k_buf,
                                                 DAT::tdual_int_1d k_sendlist,
                                                 DAT::tdual_int_1d k_copylist,
                                                 ExecutionSpace space)
{
  // 3 comp of x, 3 comp of v, 1 tag, 1 type, 1 mask, 1 image, 1 molecule, 3 nspecial,
  // maxspecial special, 1 num_bond, bond_per_atom bond_type, bond_per_atom bond_atom,
  // 1 num_angle, angle_per_atom angle_type, angle_per_atom angle_atom1, angle_atom2,
  // and angle_atom3
  // 1 num_dihedral, dihedral_per_atom dihedral_type, 4*dihedral_per_atom
  // 1 num_improper, 5*improper_per_atom
  // 1 to store buffer length

  size_exchange = 19+atom->maxspecial+2*atom->bond_per_atom+4*atom->angle_per_atom+
      5*atom->dihedral_per_atom + 5*atom->improper_per_atom;

  if (nsend > (int) (k_buf.h_view.extent(0)*
              k_buf.h_view.extent(1))/size_exchange) {
    int newsize = nsend*size_exchange/k_buf.h_view.extent(1)+1;
    k_buf.resize(newsize,k_buf.h_view.extent(1));
  }
  if (space == HostKK) {
    AtomVecMolecularKokkos_PackExchangeFunctor<LMPHostType>
      f(atomKK,k_buf,k_sendlist,k_copylist);
    Kokkos::parallel_for(nsend,f);
    return nsend*size_exchange;
  } else {
    AtomVecMolecularKokkos_PackExchangeFunctor<LMPDeviceType>
      f(atomKK,k_buf,k_sendlist,k_copylist);
    Kokkos::parallel_for(nsend,f);
    return nsend*size_exchange;
  }
}

/* ---------------------------------------------------------------------- */

template<class DeviceType,int OUTPUT_INDICES>
struct AtomVecMolecularKokkos_UnpackExchangeFunctor {
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;
  typename AT::t_kkfloat_1d_3_lr _x;
  typename AT::t_kkfloat_1d_3 _v;
  typename AT::t_tagint_1d _tag;
  typename AT::t_int_1d _type;
  typename AT::t_int_1d _mask;
  typename AT::t_imageint_1d _image;
  typename AT::t_tagint_1d _molecule;
  typename AT::t_int_2d _nspecial;
  typename AT::t_tagint_2d _special;
  typename AT::t_int_1d _num_bond;
  typename AT::t_int_2d _bond_type;
  typename AT::t_tagint_2d _bond_atom;
  typename AT::t_int_1d _num_angle;
  typename AT::t_int_2d _angle_type;
  typename AT::t_tagint_2d _angle_atom1,_angle_atom2,_angle_atom3;
  typename AT::t_int_1d _num_dihedral;
  typename AT::t_int_2d _dihedral_type;
  typename AT::t_tagint_2d _dihedral_atom1,_dihedral_atom2,
    _dihedral_atom3,_dihedral_atom4;
  typename AT::t_int_1d _num_improper;
  typename AT::t_int_2d _improper_type;
  typename AT::t_tagint_2d _improper_atom1,_improper_atom2,
    _improper_atom3,_improper_atom4;

  typename AT::t_double_2d_lr_um _buf;
  typename AT::t_int_1d _nlocal;
  typename AT::t_int_1d _indices;
  int _dim;
  double _lo,_hi;
  int _size_exchange;

  AtomVecMolecularKokkos_UnpackExchangeFunctor(
    const AtomKokkos* atom,
    const DAT::tdual_double_2d_lr buf,
    DAT::tdual_int_1d nlocal,
    DAT::tdual_int_1d indices,
    int dim, double lo, double hi):
      _x(atom->k_x.view<DeviceType>()),
      _v(atom->k_v.view<DeviceType>()),
      _tag(atom->k_tag.view<DeviceType>()),
      _type(atom->k_type.view<DeviceType>()),
      _mask(atom->k_mask.view<DeviceType>()),
      _image(atom->k_image.view<DeviceType>()),
      _molecule(atom->k_molecule.view<DeviceType>()),
      _nspecial(atom->k_nspecial.view<DeviceType>()),
      _special(atom->k_special.view<DeviceType>()),
      _num_bond(atom->k_num_bond.view<DeviceType>()),
      _bond_type(atom->k_bond_type.view<DeviceType>()),
      _bond_atom(atom->k_bond_atom.view<DeviceType>()),
      _num_angle(atom->k_num_angle.view<DeviceType>()),
      _angle_type(atom->k_angle_type.view<DeviceType>()),
      _angle_atom1(atom->k_angle_atom1.view<DeviceType>()),
      _angle_atom2(atom->k_angle_atom2.view<DeviceType>()),
      _angle_atom3(atom->k_angle_atom3.view<DeviceType>()),
      _num_dihedral(atom->k_num_dihedral.view<DeviceType>()),
      _dihedral_type(atom->k_dihedral_type.view<DeviceType>()),
      _dihedral_atom1(atom->k_dihedral_atom1.view<DeviceType>()),
      _dihedral_atom2(atom->k_dihedral_atom2.view<DeviceType>()),
      _dihedral_atom3(atom->k_dihedral_atom3.view<DeviceType>()),
      _dihedral_atom4(atom->k_dihedral_atom4.view<DeviceType>()),
      _num_improper(atom->k_num_improper.view<DeviceType>()),
      _improper_type(atom->k_improper_type.view<DeviceType>()),
      _improper_atom1(atom->k_improper_atom1.view<DeviceType>()),
      _improper_atom2(atom->k_improper_atom2.view<DeviceType>()),
      _improper_atom3(atom->k_improper_atom3.view<DeviceType>()),
      _improper_atom4(atom->k_improper_atom4.view<DeviceType>()),
      _nlocal(nlocal.template view<DeviceType>()),
      _indices(indices.template view<DeviceType>()),
      _dim(dim),_lo(lo),_hi(hi),_size_exchange(atom->avecKK->size_exchange) {
        const int maxsendlist = (buf.template view<DeviceType>().extent(0)*
                                 buf.template view<DeviceType>().extent(1))/_size_exchange;
        buffer_view<DeviceType>(_buf,buf,maxsendlist,_size_exchange);
  }

  KOKKOS_INLINE_FUNCTION
  void operator() (const int &myrecv) const {
    double x = _buf(myrecv,_dim+1);
    int i = -1;
    if (x >= _lo && x < _hi) {
      i = Kokkos::atomic_fetch_add(&_nlocal(0),1);
      int m = 1;
      _x(i,0) = _buf(myrecv,m++);
      _x(i,1) = _buf(myrecv,m++);
      _x(i,2) = _buf(myrecv,m++);
      _v(i,0) = _buf(myrecv,m++);
      _v(i,1) = _buf(myrecv,m++);
      _v(i,2) = _buf(myrecv,m++);
      _tag(i) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      _type(i) = (int) d_ubuf(_buf(myrecv,m++)).i;
      _mask(i) = (int) d_ubuf(_buf(myrecv,m++)).i;
      _image(i) = (imageint) d_ubuf(_buf(myrecv,m++)).i;

      _molecule(i) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      _num_bond(i) = (int) d_ubuf(_buf(myrecv,m++)).i;
      int k;
      for (k = 0; k < _num_bond(i); k++) {
        _bond_type(i,k) = (int) d_ubuf(_buf(myrecv,m++)).i;
        _bond_atom(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      }
      _num_angle(i) =  (int) d_ubuf(_buf(myrecv,m++)).i;
      for (k = 0; k < _num_angle(i); k++) {
        _angle_type(i,k) = (int) d_ubuf(_buf(myrecv,m++)).i;
        _angle_atom1(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _angle_atom2(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _angle_atom3(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      }
      _num_dihedral(i) = d_ubuf(_buf(myrecv,m++)).i;
      for (k = 0; k < _num_dihedral(i); k++) {
        _dihedral_type(i,k) = (int) d_ubuf(_buf(myrecv,m++)).i;
        _dihedral_atom1(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _dihedral_atom2(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _dihedral_atom3(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _dihedral_atom4(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      }
      _num_improper(i) =  (int) d_ubuf(_buf(myrecv,m++)).i;
      for (k = 0; k < (int) _num_improper(i); k++) {
        _improper_type(i,k) = (int) d_ubuf(_buf(myrecv,m++)).i;
        _improper_atom1(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _improper_atom2(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _improper_atom3(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
        _improper_atom4(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
      }
      _nspecial(i,0) = (int) d_ubuf(_buf(myrecv,m++)).i;
      _nspecial(i,1) = (int) d_ubuf(_buf(myrecv,m++)).i;
      _nspecial(i,2) = (int) d_ubuf(_buf(myrecv,m++)).i;
      for (k = 0; k < _nspecial(i,2); k++)
        _special(i,k) = (tagint) d_ubuf(_buf(myrecv,m++)).i;
    }
    if (OUTPUT_INDICES)
      _indices(myrecv) = i;
  }
};

/* ---------------------------------------------------------------------- */
int AtomVecMolecularKokkos::unpack_exchange_kokkos(DAT::tdual_double_2d_lr &k_buf, int nrecv, int nlocal,
                                                   int dim, double lo, double hi, ExecutionSpace space,
                                                   DAT::tdual_int_1d &k_indices)
{
  while (nlocal + nrecv/size_exchange >= nmax) grow(0);

  if (space == HostKK) {
    if (k_indices.h_view.data()) {
      k_count.h_view(0) = nlocal;
      AtomVecMolecularKokkos_UnpackExchangeFunctor<LMPHostType,1>
        f(atomKK,k_buf,k_count,k_indices,dim,lo,hi);
      Kokkos::parallel_for(nrecv/size_exchange,f);
    } else {
      k_count.h_view(0) = nlocal;
      AtomVecMolecularKokkos_UnpackExchangeFunctor<LMPHostType,0>
        f(atomKK,k_buf,k_count,k_indices,dim,lo,hi);
      Kokkos::parallel_for(nrecv/size_exchange,f);
    }
  } else {
    if (k_indices.h_view.data()) {
      k_count.h_view(0) = nlocal;
      k_count.modify_host();
      k_count.sync_device();
      AtomVecMolecularKokkos_UnpackExchangeFunctor<LMPDeviceType,1>
        f(atomKK,k_buf,k_count,k_indices,dim,lo,hi);
      Kokkos::parallel_for(nrecv/size_exchange,f);
      k_count.modify_device();
      k_count.sync_host();
    } else {
      k_count.h_view(0) = nlocal;
      k_count.modify_host();
      k_count.sync_device();
      AtomVecMolecularKokkos_UnpackExchangeFunctor<LMPDeviceType,0>
        f(atomKK,k_buf,k_count,k_indices,dim,lo,hi);
      Kokkos::parallel_for(nrecv/size_exchange,f);
      k_count.modify_device();
      k_count.sync_host();
    }
  }

  return k_count.h_view(0);
}

/* ---------------------------------------------------------------------- */

void AtomVecMolecularKokkos::sync(ExecutionSpace space, unsigned int mask)
{
  if (space == Device) {
    if (mask & X_MASK) atomKK->k_x.sync_device();
    if (mask & V_MASK) atomKK->k_v.sync_device();
    if (mask & F_MASK) atomKK->k_f.sync_device();
    if (mask & TAG_MASK) atomKK->k_tag.sync_device();
    if (mask & TYPE_MASK) atomKK->k_type.sync_device();
    if (mask & MASK_MASK) atomKK->k_mask.sync_device();
    if (mask & IMAGE_MASK) atomKK->k_image.sync_device();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.sync_device();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.sync_device();
      atomKK->k_special.sync_device();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.sync_device();
      atomKK->k_bond_type.sync_device();
      atomKK->k_bond_atom.sync_device();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.sync_device();
      atomKK->k_angle_type.sync_device();
      atomKK->k_angle_atom1.sync_device();
      atomKK->k_angle_atom2.sync_device();
      atomKK->k_angle_atom3.sync_device();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.sync_device();
      atomKK->k_dihedral_type.sync_device();
      atomKK->k_dihedral_atom1.sync_device();
      atomKK->k_dihedral_atom2.sync_device();
      atomKK->k_dihedral_atom3.sync_device();
      atomKK->k_dihedral_atom4.sync_device();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.sync_device();
      atomKK->k_improper_type.sync_device();
      atomKK->k_improper_atom1.sync_device();
      atomKK->k_improper_atom2.sync_device();
      atomKK->k_improper_atom3.sync_device();
      atomKK->k_improper_atom4.sync_device();
    }
  } else if (space == Host) {
    if (mask & X_MASK) atomKK->k_x.sync_host();
    if (mask & V_MASK) atomKK->k_v.sync_host();
    if (mask & F_MASK) atomKK->k_f.sync_host();
    if (mask & TAG_MASK) atomKK->k_tag.sync_host();
    if (mask & TYPE_MASK) atomKK->k_type.sync_host();
    if (mask & MASK_MASK) atomKK->k_mask.sync_host();
    if (mask & IMAGE_MASK) atomKK->k_image.sync_host();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.sync_host();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.sync_host();
      atomKK->k_special.sync_host();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.sync_host();
      atomKK->k_bond_type.sync_host();
      atomKK->k_bond_atom.sync_host();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.sync_host();
      atomKK->k_angle_type.sync_host();
      atomKK->k_angle_atom1.sync_host();
      atomKK->k_angle_atom2.sync_host();
      atomKK->k_angle_atom3.sync_host();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.sync_host();
      atomKK->k_dihedral_type.sync_host();
      atomKK->k_dihedral_atom1.sync_host();
      atomKK->k_dihedral_atom2.sync_host();
      atomKK->k_dihedral_atom3.sync_host();
      atomKK->k_dihedral_atom4.sync_host();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.sync_host();
      atomKK->k_improper_type.sync_host();
      atomKK->k_improper_atom1.sync_host();
      atomKK->k_improper_atom2.sync_host();
      atomKK->k_improper_atom3.sync_host();
      atomKK->k_improper_atom4.sync_host();
    }
  } else if (space == HostKK) {
    if (mask & X_MASK) atomKK->k_x.sync_hostkk();
    if (mask & V_MASK) atomKK->k_v.sync_hostkk();
    if (mask & F_MASK) atomKK->k_f.sync_hostkk();
    if (mask & TAG_MASK) atomKK->k_tag.sync_host();
    if (mask & TYPE_MASK) atomKK->k_type.sync_host();
    if (mask & MASK_MASK) atomKK->k_mask.sync_host();
    if (mask & IMAGE_MASK) atomKK->k_image.sync_host();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.sync_host();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.sync_hostkk();
      atomKK->k_special.sync_hostkk();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.sync_host();
      atomKK->k_bond_type.sync_hostkk();
      atomKK->k_bond_atom.sync_hostkk();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.sync_host();
      atomKK->k_angle_type.sync_hostkk();
      atomKK->k_angle_atom1.sync_hostkk();
      atomKK->k_angle_atom2.sync_hostkk();
      atomKK->k_angle_atom3.sync_hostkk();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.sync_host();
      atomKK->k_dihedral_type.sync_hostkk();
      atomKK->k_dihedral_atom1.sync_hostkk();
      atomKK->k_dihedral_atom2.sync_hostkk();
      atomKK->k_dihedral_atom3.sync_hostkk();
      atomKK->k_dihedral_atom4.sync_hostkk();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.sync_host();
      atomKK->k_improper_type.sync_hostkk();
      atomKK->k_improper_atom1.sync_hostkk();
      atomKK->k_improper_atom2.sync_hostkk();
      atomKK->k_improper_atom3.sync_hostkk();
      atomKK->k_improper_atom4.sync_hostkk();
    }
  }
}

void AtomVecMolecularKokkos::sync_pinned(ExecutionSpace space, unsigned int mask, int async_flag)
{
  if (space == Device) {
    if ((mask & X_MASK) && atomKK->k_x.need_sync_device())
      perform_pinned_copy_transform<DAT::ttransform_kkfloat_1d_3_lr>(atomKK->k_x,space,async_flag);
    if ((mask & V_MASK) && atomKK->k_v.need_sync_device())
      perform_pinned_copy_transform<DAT::ttransform_kkfloat_1d_3>(atomKK->k_v,space,async_flag);
    if ((mask & F_MASK) && atomKK->k_f.need_sync_device())
      perform_pinned_copy_transform<DAT::ttransform_kkacc_1d_3>(atomKK->k_f,space,async_flag);
    if ((mask & TAG_MASK) && atomKK->k_tag.need_sync_device())
      perform_pinned_copy<DAT::tdual_tagint_1d>(atomKK->k_tag,space,async_flag);
    if ((mask & TYPE_MASK) && atomKK->k_type.need_sync_device())
      perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_type,space,async_flag);
    if ((mask & MASK_MASK) && atomKK->k_mask.need_sync_device())
      perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_mask,space,async_flag);
    if ((mask & IMAGE_MASK) && atomKK->k_image.need_sync_device())
      perform_pinned_copy<DAT::tdual_imageint_1d>(atomKK->k_image,space,async_flag);
    if ((mask & MOLECULE_MASK) && atomKK->k_molecule.need_sync_device())
      perform_pinned_copy<DAT::tdual_tagint_1d>(atomKK->k_molecule,space,async_flag);
    if (mask & SPECIAL_MASK) {
      if (atomKK->k_nspecial.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_nspecial,space,async_flag);
      if (atomKK->k_special.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_special,space,async_flag);
    }
    if (mask & BOND_MASK) {
      if (atomKK->k_num_bond.need_sync_device())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_bond,space,async_flag);
      if (atomKK->k_bond_type.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_bond_type,space,async_flag);
      if (atomKK->k_bond_atom.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_bond_atom,space,async_flag);
    }
    if (mask & ANGLE_MASK) {
      if (atomKK->k_num_angle.need_sync_device())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_angle,space,async_flag);
      if (atomKK->k_angle_type.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_angle_type,space,async_flag);
      if (atomKK->k_angle_atom1.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom1,space,async_flag);
      if (atomKK->k_angle_atom2.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom2,space,async_flag);
      if (atomKK->k_angle_atom3.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom3,space,async_flag);
    }
    if (mask & DIHEDRAL_MASK) {
      if (atomKK->k_num_dihedral.need_sync_device())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_dihedral,space,async_flag);
      if (atomKK->k_dihedral_type.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_dihedral_type,space,async_flag);
      if (atomKK->k_dihedral_atom1.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom1,space,async_flag);
      if (atomKK->k_dihedral_atom2.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom2,space,async_flag);
      if (atomKK->k_dihedral_atom3.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom3,space,async_flag);
    }
    if (mask & IMPROPER_MASK) {
      if (atomKK->k_num_improper.need_sync_device())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_improper,space,async_flag);
      if (atomKK->k_improper_type.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_improper_type,space,async_flag);
      if (atomKK->k_improper_atom1.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom1,space,async_flag);
      if (atomKK->k_improper_atom2.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom2,space,async_flag);
      if (atomKK->k_improper_atom3.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom3,space,async_flag);
      if (atomKK->k_improper_atom4.need_sync_device())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom4,space,async_flag);
    }
  } else {
    if ((mask & X_MASK) && atomKK->k_x.need_sync_host())
      perform_pinned_copy_transform<DAT::ttransform_kkfloat_1d_3_lr>(atomKK->k_x,space,async_flag);
    if ((mask & V_MASK) && atomKK->k_v.need_sync_host())
      perform_pinned_copy_transform<DAT::ttransform_kkfloat_1d_3>(atomKK->k_v,space,async_flag);
    if ((mask & F_MASK) && atomKK->k_f.need_sync_host())
      perform_pinned_copy_transform<DAT::ttransform_kkacc_1d_3>(atomKK->k_f,space,async_flag);
    if ((mask & TAG_MASK) && atomKK->k_tag.need_sync_host())
      perform_pinned_copy<DAT::tdual_tagint_1d>(atomKK->k_tag,space,async_flag);
    if ((mask & TYPE_MASK) && atomKK->k_type.need_sync_host())
      perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_type,space,async_flag);
    if ((mask & MASK_MASK) && atomKK->k_mask.need_sync_host())
      perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_mask,space,async_flag);
    if ((mask & IMAGE_MASK) && atomKK->k_image.need_sync_host())
      perform_pinned_copy<DAT::tdual_imageint_1d>(atomKK->k_image,space,async_flag);
    if ((mask & MOLECULE_MASK) && atomKK->k_molecule.need_sync_host())
      perform_pinned_copy<DAT::tdual_tagint_1d>(atomKK->k_molecule,space,async_flag);
    if (mask & SPECIAL_MASK) {
      if (atomKK->k_nspecial.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_nspecial,space,async_flag);
      if (atomKK->k_special.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_special,space,async_flag);
    }
    if (mask & BOND_MASK) {
      if (atomKK->k_num_bond.need_sync_host())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_bond,space,async_flag);
      if (atomKK->k_bond_type.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_bond_type,space,async_flag);
      if (atomKK->k_bond_atom.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_bond_atom,space,async_flag);
    }
    if (mask & ANGLE_MASK) {
      if (atomKK->k_num_angle.need_sync_host())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_angle,space,async_flag);
      if (atomKK->k_angle_type.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_angle_type,space,async_flag);
      if (atomKK->k_angle_atom1.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom1,space,async_flag);
      if (atomKK->k_angle_atom2.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom2,space,async_flag);
      if (atomKK->k_angle_atom3.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_angle_atom3,space,async_flag);
    }
    if (mask & DIHEDRAL_MASK) {
      if (atomKK->k_num_dihedral.need_sync_host())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_dihedral,space,async_flag);
      if (atomKK->k_dihedral_type.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_dihedral_type,space,async_flag);
      if (atomKK->k_dihedral_atom1.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom1,space,async_flag);
      if (atomKK->k_dihedral_atom2.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom2,space,async_flag);
      if (atomKK->k_dihedral_atom3.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom3,space,async_flag);
      if (atomKK->k_dihedral_atom4.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_dihedral_atom4,space,async_flag);
    }
    if (mask & IMPROPER_MASK) {
      if (atomKK->k_num_improper.need_sync_host())
        perform_pinned_copy<DAT::tdual_int_1d>(atomKK->k_num_improper,space,async_flag);
      if (atomKK->k_improper_type.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_int_2d>(atomKK->k_improper_type,space,async_flag);
      if (atomKK->k_improper_atom1.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom1,space,async_flag);
      if (atomKK->k_improper_atom2.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom2,space,async_flag);
      if (atomKK->k_improper_atom3.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom3,space,async_flag);
      if (atomKK->k_improper_atom4.need_sync_host())
        perform_pinned_copy_transform<DAT::ttransform_tagint_2d>(atomKK->k_improper_atom4,space,async_flag);
    }
  }
}

/* ---------------------------------------------------------------------- */

void AtomVecMolecularKokkos::modified(ExecutionSpace space, unsigned int mask)
{
  if (space == Device) {
    if (mask & X_MASK) atomKK->k_x.modify_device();
    if (mask & V_MASK) atomKK->k_v.modify_device();
    if (mask & F_MASK) atomKK->k_f.modify_device();
    if (mask & TAG_MASK) atomKK->k_tag.modify_device();
    if (mask & TYPE_MASK) atomKK->k_type.modify_device();
    if (mask & MASK_MASK) atomKK->k_mask.modify_device();
    if (mask & IMAGE_MASK) atomKK->k_image.modify_device();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.modify_device();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.modify_device();
      atomKK->k_special.modify_device();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.modify_device();
      atomKK->k_bond_type.modify_device();
      atomKK->k_bond_atom.modify_device();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.modify_device();
      atomKK->k_angle_type.modify_device();
      atomKK->k_angle_atom1.modify_device();
      atomKK->k_angle_atom2.modify_device();
      atomKK->k_angle_atom3.modify_device();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.modify_device();
      atomKK->k_dihedral_type.modify_device();
      atomKK->k_dihedral_atom1.modify_device();
      atomKK->k_dihedral_atom2.modify_device();
      atomKK->k_dihedral_atom3.modify_device();
      atomKK->k_dihedral_atom4.modify_device();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.modify_device();
      atomKK->k_improper_type.modify_device();
      atomKK->k_improper_atom1.modify_device();
      atomKK->k_improper_atom2.modify_device();
      atomKK->k_improper_atom3.modify_device();
      atomKK->k_improper_atom4.modify_device();
    }
  } else if (space == Host) {
    if (mask & X_MASK) atomKK->k_x.modify_host();
    if (mask & V_MASK) atomKK->k_v.modify_host();
    if (mask & F_MASK) atomKK->k_f.modify_host();
    if (mask & TAG_MASK) atomKK->k_tag.modify_host();
    if (mask & TYPE_MASK) atomKK->k_type.modify_host();
    if (mask & MASK_MASK) atomKK->k_mask.modify_host();
    if (mask & IMAGE_MASK) atomKK->k_image.modify_host();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.modify_host();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.modify_host();
      atomKK->k_special.modify_host();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.modify_host();
      atomKK->k_bond_type.modify_host();
      atomKK->k_bond_atom.modify_host();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.modify_host();
      atomKK->k_angle_type.modify_host();
      atomKK->k_angle_atom1.modify_host();
      atomKK->k_angle_atom2.modify_host();
      atomKK->k_angle_atom3.modify_host();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.modify_host();
      atomKK->k_dihedral_type.modify_host();
      atomKK->k_dihedral_atom1.modify_host();
      atomKK->k_dihedral_atom2.modify_host();
      atomKK->k_dihedral_atom3.modify_host();
      atomKK->k_dihedral_atom4.modify_host();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.modify_host();
      atomKK->k_improper_type.modify_host();
      atomKK->k_improper_atom1.modify_host();
      atomKK->k_improper_atom2.modify_host();
      atomKK->k_improper_atom3.modify_host();
      atomKK->k_improper_atom4.modify_host();
    }
  } else if (space == HostKK) {
    if (mask & X_MASK) atomKK->k_x.modify_hostkk();
    if (mask & V_MASK) atomKK->k_v.modify_hostkk();
    if (mask & F_MASK) atomKK->k_f.modify_hostkk();
    if (mask & TAG_MASK) atomKK->k_tag.modify_host();
    if (mask & TYPE_MASK) atomKK->k_type.modify_host();
    if (mask & MASK_MASK) atomKK->k_mask.modify_host();
    if (mask & IMAGE_MASK) atomKK->k_image.modify_host();
    if (mask & MOLECULE_MASK) atomKK->k_molecule.modify_host();
    if (mask & SPECIAL_MASK) {
      atomKK->k_nspecial.modify_hostkk();
      atomKK->k_special.modify_hostkk();
    }
    if (mask & BOND_MASK) {
      atomKK->k_num_bond.modify_host();
      atomKK->k_bond_type.modify_hostkk();
      atomKK->k_bond_atom.modify_hostkk();
    }
    if (mask & ANGLE_MASK) {
      atomKK->k_num_angle.modify_host();
      atomKK->k_angle_type.modify_hostkk();
      atomKK->k_angle_atom1.modify_hostkk();
      atomKK->k_angle_atom2.modify_hostkk();
      atomKK->k_angle_atom3.modify_hostkk();
    }
    if (mask & DIHEDRAL_MASK) {
      atomKK->k_num_dihedral.modify_host();
      atomKK->k_dihedral_type.modify_hostkk();
      atomKK->k_dihedral_atom1.modify_hostkk();
      atomKK->k_dihedral_atom2.modify_hostkk();
      atomKK->k_dihedral_atom3.modify_hostkk();
      atomKK->k_dihedral_atom4.modify_hostkk();
    }
    if (mask & IMPROPER_MASK) {
      atomKK->k_num_improper.modify_host();
      atomKK->k_improper_type.modify_hostkk();
      atomKK->k_improper_atom1.modify_hostkk();
      atomKK->k_improper_atom2.modify_hostkk();
      atomKK->k_improper_atom3.modify_hostkk();
      atomKK->k_improper_atom4.modify_hostkk();
    }
  }
}
