/* -*- c++ -*- ----------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   LAMMPS development team: developers@lammps.org

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#ifdef COMPUTE_CLASS
// clang-format off
ComputeStyle(temp/deform/kk,ComputeTempDeformKokkos<LMPDeviceType>);
ComputeStyle(temp/deform/kk/device,ComputeTempDeformKokkos<LMPDeviceType>);
ComputeStyle(temp/deform/kk/host,ComputeTempDeformKokkos<LMPHostType>);
// clang-format on
#else

// clang-format off
#ifndef LMP_COMPUTE_TEMP_DEFORM_KOKKOS_H
#define LMP_COMPUTE_TEMP_DEFORM_KOKKOS_H

#include "compute_temp_deform.h"
#include "kokkos_few.h"
#include "kokkos_type.h"

namespace LAMMPS_NS {

template<int RMASS>
struct TagComputeTempDeformScalar{};

template<int RMASS>
struct TagComputeTempDeformVector{};

struct TagComputeTempDeformRemoveBias{};

struct TagComputeTempDeformRestoreBias{};

template<class DeviceType>
class ComputeTempDeformKokkos: public ComputeTempDeform {
 public:
  struct s_CTEMP {
    double t0, t1, t2, t3, t4, t5;
    KOKKOS_INLINE_FUNCTION
    s_CTEMP() {
      t0 = t1 = t2 = t3 = t4 = t5 = 0.0;
    }
    KOKKOS_INLINE_FUNCTION
    s_CTEMP& operator+=(const s_CTEMP &rhs) {
      t0 += rhs.t0;
      t1 += rhs.t1;
      t2 += rhs.t2;
      t3 += rhs.t3;
      t4 += rhs.t4;
      t5 += rhs.t5;
      return *this;
    }
  };

  typedef s_CTEMP CTEMP;
  typedef CTEMP value_type;
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;

  ComputeTempDeformKokkos(class LAMMPS *, int, char **);
  double compute_scalar() override;
  void compute_vector() override;
  void remove_bias_all() override;
  void remove_bias_all_kk() override;
  void restore_bias_all() override;

  template<int RMASS>
  KOKKOS_INLINE_FUNCTION
  void operator()(TagComputeTempDeformScalar<RMASS>, const int&, CTEMP&) const;

  template<int RMASS>
  KOKKOS_INLINE_FUNCTION
  void operator()(TagComputeTempDeformVector<RMASS>, const int&, CTEMP&) const;

  KOKKOS_INLINE_FUNCTION
  void operator()(TagComputeTempDeformRemoveBias, const int &i) const;

  KOKKOS_INLINE_FUNCTION
  void operator()(TagComputeTempDeformRestoreBias, const int &i) const;

 protected:
  typename AT::t_kkfloat_1d_3_lr_randomread x;
  typename AT::t_kkfloat_1d_3 v;
  typename AT::t_kkfloat_1d_3 vbiasall;
  typename AT::t_kkfloat_1d_randomread rmass;
  typename AT::t_kkfloat_1d_randomread mass;
  typename AT::t_int_1d_randomread type;
  typename AT::t_int_1d_randomread mask;

  class DomainKokkos *domainKK;

  Few<double, 6> h_rate, h_ratelo;

  };

}

#endif
#endif

